<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Organization;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class CreateOrganization extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'org:create {name} {admin-email} {admin-name?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a new organization with admin user';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $name = $this->argument('name');
        $adminEmail = $this->argument('admin-email');
        $adminName = $this->argument('admin-name') ?: 'Organization Admin';
        
        // Check if organization already exists
        if (Organization::where('name', $name)->exists()) {
            $this->error("Organization '{$name}' already exists!");
            return 1;
        }
        
        // Check if admin email already exists
        if (User::where('email', $adminEmail)->exists()) {
            $this->error("User with email '{$adminEmail}' already exists!");
            return 1;
        }
        
        $this->info("Creating organization: {$name}");
        
        // Create organization
        $organization = Organization::create([
            'name' => $name,
            'slug' => Str::slug($name),
            'status' => 'active',
            'settings' => [],
        ]);
        
        // Generate random password
        $password = Str::random(12);
        
        // Create admin user
        $adminUser = User::create([
            'name' => $adminName,
            'email' => $adminEmail,
            'password' => Hash::make($password),
            'organization_id' => $organization->id,
            'is_system_admin' => false,
        ]);
        
        // Assign organization admin role
        $adminUser->assignRole('org_admin');
        
        $this->info("✅ Organization created successfully!");
        $this->info("📋 Details:");
        $this->info("   Name: {$organization->name}");
        $this->info("   Slug: {$organization->slug}");
        $this->info("   Admin: {$adminUser->name} ({$adminUser->email})");
        $this->info("   Password: {$password}");
        $this->warn("⚠️  Please save the password - it won't be shown again!");
        
        return 0;
    }
}