<?php

namespace App\Console\Commands;

use App\Models\User;
use App\Services\OtpService;
use Illuminate\Console\Command;
use Spatie\OneTimePasswords\Models\OneTimePassword;

class GenerateOtp extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'otp:generate {email : The user email address}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate and send OTP for a user';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $email = $this->argument('email');
        
        $user = User::where('email', $email)->first();
        
        if (!$user) {
            $this->error("User not found: {$email}");
            return 1;
        }
        
        $otpService = app(OtpService::class);
        
        if (!$otpService->canResend($user)) {
            $seconds = $otpService->secondsUntilCanResend($user);
            $this->warn("Rate limited! Please wait {$seconds} seconds.");
            return 1;
        }
        
        $this->info("Generating OTP for {$user->name} ({$email})...");
        
        $result = $otpService->generateAndSendOtp($user);
        
        if ($result) {
            $otp = OneTimePassword::where('authenticatable_id', $user->id)->latest()->first();
            
            $this->line('');
            $this->info('✅ OTP generated and email sent!');
            $this->line('');
            $this->line("  Code: {$otp->password}");
            $this->line("  Expires: {$otp->expires_at->diffForHumans()}");
            $this->line('');
            $this->comment("Check email inbox for: {$email}");
            
            return 0;
        }
        
        $this->error('Failed to generate OTP');
        return 1;
    }
}
