<?php

namespace App\Console\Commands;

use App\Models\Invoice;
use App\Models\Organization;
use App\Models\SubscriptionPlan;
use App\Services\MpesaService;
use Illuminate\Console\Command;

class TestMpesaIntegration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mpesa:test {phone?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test M-Pesa STK Push integration';

    /**
     * Execute the console command.
     */
    public function handle(MpesaService $mpesaService)
    {
        $this->info('🧪 Testing M-Pesa Integration...');
        $this->newLine();

        // Get phone number
        $phone = $this->argument('phone') ?? $this->ask('Enter phone number (e.g., 254712345678)');

        if (!$phone) {
            $this->error('Phone number is required');
            return 1;
        }

        // Get or create test organization
        $organization = Organization::first();
        
        if (!$organization) {
            $this->error('No organization found. Please run database seeders first.');
            return 1;
        }

        // Get starter plan
        $plan = SubscriptionPlan::where('slug', 'starter')->first();

        if (!$plan) {
            $this->error('No subscription plan found. Please run subscription seeder first.');
            return 1;
        }

        // Create test invoice
        $invoice = Invoice::create([
            'organization_id' => $organization->id,
            'subscription_plan_id' => $plan->id,
            'invoice_number' => Invoice::generateInvoiceNumber(),
            'type' => 'subscription',
            'subtotal' => 1.00, // Test with KES 1
            'tax' => 0,
            'discount' => 0,
            'total' => 1.00,
            'status' => 'pending',
            'issue_date' => now(),
            'due_date' => now()->addDays(7),
            'line_items' => [
                [
                    'description' => 'Test Subscription Payment',
                    'amount' => 1.00,
                ],
            ],
        ]);

        $this->info("✅ Test invoice created: {$invoice->invoice_number}");
        $this->info("   Amount: KES {$invoice->total}");
        $this->newLine();

        // Initiate STK Push
        $this->info('📱 Initiating STK Push...');
        $result = $mpesaService->initiateSubscriptionPayment($invoice, $phone);

        if ($result['success']) {
            $this->info('✅ ' . $result['message']);
            $this->info("   Payment ID: {$result['payment_id']}");
            $this->info("   Checkout Request ID: {$result['checkout_request_id']}");
            $this->newLine();
            $this->info('📲 Please check your phone and enter M-Pesa PIN to complete payment');
            $this->newLine();
            
            // Wait for callback
            $this->info('⏳ Waiting for payment confirmation (30 seconds)...');
            
            $payment = \App\Models\Payment::find($result['payment_id']);
            $attempts = 0;
            $maxAttempts = 30;
            
            while ($attempts < $maxAttempts) {
                sleep(1);
                $payment->refresh();
                
                if ($payment->isCompleted()) {
                    $this->newLine();
                    $this->info('✅ Payment completed successfully!');
                    $this->info("   Receipt: {$payment->mpesa_receipt_number}");
                    $this->info("   Amount: KES {$payment->mpesa_amount}");
                    return 0;
                }
                
                if ($payment->isFailed()) {
                    $this->newLine();
                    $this->error('❌ Payment failed');
                    $this->error("   Reason: {$payment->notes}");
                    return 1;
                }
                
                $attempts++;
                $this->output->write('.');
            }
            
            $this->newLine();
            $this->newLine();
            $this->warn('⏱️  Timeout waiting for payment confirmation');
            $this->info('   Payment status: ' . $payment->status);
            $this->info('   You can check status later with: php artisan mpesa:check-status ' . $payment->id);
            
        } else {
            $this->error('❌ Failed to initiate payment');
            $this->error('   ' . $result['message']);
            
            if (isset($result['error'])) {
                $this->error('   Error: ' . $result['error']);
            }
            
            return 1;
        }

        return 0;
    }
}
