<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException;
use Throwable;

class Handler extends ExceptionHandler
{
    /**
     * The list of the inputs that are never flashed to the session on validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     */
    public function register(): void
    {
        $this->reportable(function (Throwable $e) {
            //
        });
    }

    /**
     * Render an exception into an HTTP response.
     */
    public function render($request, Throwable $e)
    {
        // Handle specific HTTP exceptions with custom views
        if ($e instanceof NotFoundHttpException) {
            return response()->view('errors.404', [
                'exception' => $e,
                'title' => 'Page Not Found',
                'message' => 'The page you are looking for could not be found.'
            ], 404);
        }

        if ($e instanceof AccessDeniedHttpException || $e instanceof AuthorizationException) {
            return response()->view('errors.403', [
                'exception' => $e,
                'title' => 'Access Denied',
                'message' => 'You do not have permission to access this resource.'
            ], 403);
        }

        if ($e instanceof TooManyRequestsHttpException) {
            return response()->view('errors.429', [
                'exception' => $e,
                'title' => 'Too Many Requests',
                'message' => 'You have made too many requests. Please try again later.'
            ], 429);
        }

        // Handle authentication exceptions
        if ($e instanceof AuthenticationException) {
            return redirect()->guest(route('login'));
        }

        // Handle model not found exceptions
        if ($e instanceof ModelNotFoundException) {
            return response()->view('errors.404', [
                'exception' => $e,
                'title' => 'Resource Not Found',
                'message' => 'The requested resource could not be found.'
            ], 404);
        }

        // Handle server errors (500)
        if ($e instanceof HttpException && $e->getStatusCode() === 500) {
            return response()->view('errors.500', [
                'exception' => $e,
                'title' => 'Server Error',
                'message' => 'Something went wrong on our end. Please try again later.'
            ], 500);
        }

        // Handle general HTTP exceptions
        if ($e instanceof HttpException) {
            $statusCode = $e->getStatusCode();
            
            if (view()->exists("errors.{$statusCode}")) {
                return response()->view("errors.{$statusCode}", [
                    'exception' => $e,
                    'title' => $this->getErrorTitle($statusCode),
                    'message' => $this->getErrorMessage($statusCode)
                ], $statusCode);
            }
        }

        return parent::render($request, $e);
    }

    /**
     * Get error title based on status code
     */
    private function getErrorTitle(int $statusCode): string
    {
        return match($statusCode) {
            400 => 'Bad Request',
            401 => 'Unauthorized',
            403 => 'Access Denied',
            404 => 'Page Not Found',
            405 => 'Method Not Allowed',
            408 => 'Request Timeout',
            422 => 'Validation Error',
            429 => 'Too Many Requests',
            500 => 'Server Error',
            502 => 'Bad Gateway',
            503 => 'Service Unavailable',
            504 => 'Gateway Timeout',
            default => 'Error'
        };
    }

    /**
     * Get error message based on status code
     */
    private function getErrorMessage(int $statusCode): string
    {
        return match($statusCode) {
            400 => 'The request could not be understood by the server.',
            401 => 'You need to log in to access this resource.',
            403 => 'You do not have permission to access this resource.',
            404 => 'The page you are looking for could not be found.',
            405 => 'The request method is not allowed for this resource.',
            408 => 'The request took too long to process.',
            422 => 'The provided data is invalid.',
            429 => 'You have made too many requests. Please try again later.',
            500 => 'Something went wrong on our end. Please try again later.',
            502 => 'The server received an invalid response.',
            503 => 'The service is temporarily unavailable.',
            504 => 'The server took too long to respond.',
            default => 'An unexpected error occurred.'
        };
    }
}