<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Services\AuditService;
use App\Exports\AuditLogsExport;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class AuditLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_audit_logs');
    }

    /**
     * Display audit logs with filtering and pagination
     */
    public function index(Request $request): View|JsonResponse
    {
        $query = AuditLog::with(['user', 'organization'])
            ->latest();

        // Apply organization filter based on user role and permissions
        if (Auth::user()->hasPermission('view_system_audit_logs')) {
            // System admin can see all audit logs
            // No additional filtering needed
        } elseif (Auth::user()->hasPermission('view_organization_audit_logs')) {
            // Organization users can only see their organization's logs
            $query->forOrganization(Auth::user()->organization_id);
        } else {
            // Fallback: no access
            abort(403, 'Unauthorized to view audit logs');
        }

        // Apply filters
        $this->applyFilters($query, $request);

        // Handle AJAX requests for real-time filtering
        if ($request->ajax()) {
            $logs = $query->paginate(25);
            return response()->json([
                'html' => view('admin.audit-logs.table', compact('logs'))->render(),
                'pagination' => $logs->links()->render(),
                'total' => $logs->total(),
            ]);
        }

        $logs = $query->paginate(25);
        $statistics = AuditService::getStatistics(
            Auth::user()->hasRole('system_admin') ? null : Auth::user()->organization_id
        );

        return view('admin.audit-logs.index', compact('logs', 'statistics'));
    }

    /**
     * Show detailed audit log entry
     */
    public function show(AuditLog $auditLog): View
    {
        // Check if user can view this audit log
        if (!Auth::user()->hasPermission('view_system_audit_logs') && 
            $auditLog->organization_id !== Auth::user()->organization_id) {
            abort(403);
        }

        return view('admin.audit-logs.show', compact('auditLog'));
    }

    /**
     * Export audit logs
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:csv,xlsx',
            'from_date' => 'nullable|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
        ]);

        $query = AuditLog::with(['user', 'organization']);

        // Apply organization filter
        if (!Auth::user()->hasPermission('view_system_audit_logs')) {
            $query->forOrganization(Auth::user()->organization_id);
        }

        // Apply date range filter
        if ($request->from_date && $request->to_date) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        $this->applyFilters($query, $request);

        // Limit export size for performance
        $maxRecords = config('audit.export.max_records', 50000);
        if ((clone $query)->count() > $maxRecords) {
            return back()->withErrors([
                'export' => "Export limited to {$maxRecords} records. Please narrow your date range."
            ]);
        }

        $filename = 'audit_logs_' . now()->format('Y-m-d_H-i-s') . '.' . $request->format;

        return Excel::download(
            new AuditLogsExport($query),
            $filename
        );
    }

    /**
     * Get audit statistics for dashboard
     */
    public function statistics(Request $request): JsonResponse
    {
        $days = $request->get('days', 30);
        $organizationId = Auth::user()->hasPermission('view_system_audit_logs') 
            ? $request->get('organization_id') 
            : Auth::user()->organization_id;

        $statistics = AuditService::getStatistics($organizationId, $days);

        return response()->json($statistics);
    }

    /**
     * Apply filters to the audit log query
     */
    private function applyFilters($query, Request $request): void
    {
        if ($request->filled('user_id')) {
            $query->byUser($request->user_id);
        }

        if ($request->filled('event_type')) {
            $query->byEventType($request->event_type);
        }

        if ($request->filled('category')) {
            $query->byCategory($request->category);
        }

        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        if ($request->filled('search')) {
            $query->search($request->search);
        }

        if ($request->filled('auditable_type')) {
            $query->where('auditable_type', 'like', '%' . $request->auditable_type . '%');
        }
    }
}