<?php

namespace App\Http\Controllers;

use App\Models\Application;
use App\Models\ApplicationMessage;
use App\Mail\NewMessageFromHR;
use App\Mail\NewMessageFromCandidate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class ApplicationMessageController extends Controller
{
    /**
     * Show messages for HR (authenticated)
     */
    public function index(Application $application)
    {
        // Check authentication
        if (!auth()->check()) {
            abort(403, 'Unauthorized');
        }

        // Check if user can access this application (tenant scoping)
        if (!$application->belongsToCurrentOrganization()) {
            abort(403, 'Access denied.');
        }

        $messages = $application->messages()
            ->with('sender:id,name')
            ->orderBy('created_at', 'asc')
            ->get();

        // Mark all candidate messages as read by HR
        $application->messages()
            ->where('sender_type', 'candidate')
            ->where('read_by_hr', false)
            ->update(['read_by_hr' => true, 'read_at' => now()]);

        return view('recruitment.applications.messages', compact('application', 'messages'));
    }

    /**
     * HR sends message to candidate
     */
    public function store(Request $request, Application $application)
    {
        // Check authentication
        if (!auth()->check()) {
            abort(403, 'Unauthorized');
        }

        // Check if user can access this application (tenant scoping)
        if (!$application->belongsToCurrentOrganization()) {
            abort(403, 'Access denied.');
        }

        $validated = $request->validate([
            'message' => 'required|string|max:2000',
        ]);

        $message = ApplicationMessage::create([
            'application_id' => $application->id,
            'sender_type' => 'hr',
            'sender_id' => auth()->id(),
            'message' => $validated['message'],
            'read_by_hr' => true,
        ]);

        // Send email to candidate
        try {
            Mail::to($application->email)->send(new NewMessageFromHR($application, $message));
        } catch (\Exception $e) {
            \Log::error('Failed to send message email: ' . $e->getMessage());
        }

        return redirect()->back()->with('success', 'Message sent successfully!');
    }

    /**
     * Show messages for candidate (no auth)
     */
    public function candidateIndex($token)
    {
        $application = Application::where('tracking_token', $token)
            ->with(['jobPosting:id,title,organization_id', 'jobPosting.organization:id,name'])
            ->firstOrFail();

        $messages = $application->messages()
            ->with('sender:id,name')
            ->orderBy('created_at', 'asc')
            ->get();

        // Mark all HR messages as read by candidate
        $application->messages()
            ->where('sender_type', 'hr')
            ->where('read_by_candidate', false)
            ->update(['read_by_candidate' => true, 'read_at' => now()]);

        return view('candidate.portal.messages', compact('application', 'messages'));
    }

    /**
     * Candidate sends message
     */
    public function candidateStore(Request $request, $token)
    {
        $application = Application::where('tracking_token', $token)->firstOrFail();

        $validated = $request->validate([
            'message' => 'required|string|max:2000',
        ]);

        $message = ApplicationMessage::create([
            'application_id' => $application->id,
            'sender_type' => 'candidate',
            'message' => $validated['message'],
            'read_by_candidate' => true,
        ]);

        // Send email to HR
        try {
            $hrEmails = $application->jobPosting->organization->hr_email ?? config('mail.from.address');
            Mail::to($hrEmails)->send(new NewMessageFromCandidate($application, $message));
        } catch (\Exception $e) {
            \Log::error('Failed to send message email: ' . $e->getMessage());
        }

        return redirect()->back()->with('success', 'Message sent successfully!');
    }
}
