<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Services\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmailVerificationController extends Controller
{
    protected $otpService;

    public function __construct(OtpService $otpService)
    {
        $this->middleware('auth');
        $this->otpService = $otpService;
    }

    /**
     * Show the email verification page
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function show()
    {
        $user = Auth::user();

        // If email already verified, redirect to onboarding
        if ($user->hasVerifiedEmail() && !$user->must_verify_email) {
            return redirect()->route('onboarding.index');
        }

        // Send OTP if not already sent
        if (!$user->otp_code || $user->otp_expires_at->isPast()) {
            $this->otpService->generateAndSendOtp($user);
        }

        $canResend = $this->otpService->canResend($user);
        $secondsUntilResend = $this->otpService->secondsUntilCanResend($user);

        return view('auth.verify-email', compact('user', 'canResend', 'secondsUntilResend'));
    }

    /**
     * Verify the OTP code
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function verify(Request $request)
    {
        $request->validate([
            'otp_code' => ['required', 'string', 'size:6'],
        ]);

        $user = Auth::user();

        if ($this->otpService->validateOtp($user, $request->otp_code)) {
            // Update organization onboarding step
            if ($user->organization) {
                $user->organization->update([
                    'onboarding_step' => 'email_verified',
                ]);
            }

            return redirect()
                ->route('onboarding.index')
                ->with('success', 'Email verified successfully! Let\'s complete your setup.');
        }

        // Check if too many attempts
        if ($user->otp_attempts >= 5) {
            return back()->withErrors([
                'otp_code' => 'Too many failed attempts. Please request a new OTP.',
            ]);
        }

        return back()->withErrors([
            'otp_code' => 'Invalid OTP code. Please try again.',
        ]);
    }

    /**
     * Resend OTP
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function resend()
    {
        $user = Auth::user();

        if (!$this->otpService->canResend($user)) {
            $seconds = $this->otpService->secondsUntilCanResend($user);
            return back()->with('error', "Please wait {$seconds} seconds before requesting a new OTP.");
        }

        if ($this->otpService->generateAndSendOtp($user)) {
            return back()->with('success', 'A new OTP has been sent to your email.');
        }

        return back()->with('error', 'Failed to send OTP. Please try again.');
    }
}
