<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class LoginController extends Controller
{
    public function showLoginForm()
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }

        return view('auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:6',
        ]);

        // Rate limiting: 5 attempts per 15 minutes per IP
        $key = 'login-attempts-' . $request->ip();
        $attempts = cache()->get($key, 0);
        
        if ($attempts >= 5) {
            $lockoutTime = cache()->get($key . '-lockout', now());
            if (now()->lessThan($lockoutTime)) {
                return back()->withErrors([
                    'email' => 'Too many login attempts. Please try again in 15 minutes.',
                ])->withInput($request->only('email'));
            }
        }

        // Check if user exists and is not locked
        $user = \App\Models\User::where('email', $credentials['email'])->first();
        if ($user && $user->locked_until && now()->lessThan($user->locked_until)) {
            return back()->withErrors([
                'email' => 'Account is temporarily locked due to suspicious activity.',
            ])->withInput($request->only('email'));
        }

        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $request->session()->regenerate();
            
            // Clear failed attempts on successful login
            cache()->forget($key);
            cache()->forget($key . '-lockout');
            
            $user = Auth::user();
            
            // Check if 2FA is enabled
            if ($user->two_factor_secret && $user->two_factor_confirmed_at) {
                // Store user ID and remember preference in session for 2FA
                session([
                    '2fa_user_id' => $user->id,
                    '2fa_remember' => $request->filled('remember')
                ]);
                
                // Logout temporarily until 2FA is verified
                Auth::logout();
                
                return redirect()->route('two-factor.challenge');
            }
            
            // Update last login timestamp
            $user->update(['last_login_at' => now()]);
            
            // Log successful login
            Log::info('User logged in', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);

            return redirect()->intended(route('dashboard'))
                ->with('success', 'Welcome back, ' . $user->name . '!');
        }

        // Increment failed attempts
        $attempts++;
        cache()->put($key, $attempts, now()->addMinutes(15));
        
        // Lock account after 10 failed attempts
        if ($user && $attempts >= 10) {
            $user->update(['locked_until' => now()->addHours(1)]);
            cache()->put($key . '-lockout', now()->addMinutes(15), now()->addMinutes(15));
        }

        // Log failed login attempt
        if ($user) {
            Log::warning('Failed login attempt', [
                'user_id' => $user->id,
                'email' => $user->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'attempts' => $attempts
            ]);
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->withInput($request->only('email'));
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')
            ->with('success', 'You have been logged out successfully.');
    }
}