<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Services\TwoFactorAuthService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TwoFactorController extends Controller
{
    protected $twoFactorService;

    public function __construct(TwoFactorAuthService $twoFactorService)
    {
        $this->twoFactorService = $twoFactorService;
    }

    /**
     * Show 2FA setup page
     */
    public function show()
    {
        $user = Auth::user();
        $isEnabled = $this->twoFactorService->isEnabled($user);

        return view('auth.two-factor', compact('user', 'isEnabled'));
    }

    /**
     * Enable 2FA for the user
     */
    public function enable(Request $request)
    {
        $user = Auth::user();
        
        if ($this->twoFactorService->isEnabled($user)) {
            return back()->with('error', '2FA is already enabled for your account.');
        }

        $twoFactorData = $this->twoFactorService->enableTwoFactor($user);

        return view('auth.two-factor-setup', [
            'secret' => $twoFactorData['secret'],
            'recovery_codes' => $twoFactorData['recovery_codes'],
            'qr_code_url' => $twoFactorData['qr_code_url']
        ]);
    }

    /**
     * Disable 2FA for the user
     */
    public function disable(Request $request)
    {
        $request->validate([
            'password' => 'required|current_password',
        ]);

        $user = Auth::user();
        $this->twoFactorService->disableTwoFactor($user);

        return back()->with('success', '2FA has been disabled for your account.');
    }

    /**
     * Show 2FA challenge page
     */
    public function challenge()
    {
        if (!session('2fa_user_id')) {
            return redirect()->route('login');
        }

        return view('auth.two-factor-challenge');
    }

    /**
     * Verify 2FA code
     */
    public function verify(Request $request)
    {
        $request->validate([
            'code' => 'required|string|min:6|max:10',
        ]);

        $userId = session('2fa_user_id');
        if (!$userId) {
            return redirect()->route('login')->with('error', 'Session expired. Please login again.');
        }

        $user = \App\Models\User::find($userId);
        if (!$user) {
            return redirect()->route('login')->with('error', 'User not found.');
        }

        if ($this->twoFactorService->verifyCode($user, $request->code)) {
            // Complete the login process
            Auth::login($user, session('2fa_remember', false));
            
            // Clear 2FA session data
            session()->forget(['2fa_user_id', '2fa_remember']);
            
            return redirect()->intended(route('dashboard'))
                ->with('success', 'Welcome back, ' . $user->name . '!');
        }

        return back()->withErrors([
            'code' => 'The provided 2FA code is invalid.',
        ]);
    }

    /**
     * Generate new recovery codes
     */
    public function generateRecoveryCodes(Request $request)
    {
        $request->validate([
            'password' => 'required|current_password',
        ]);

        $user = Auth::user();
        
        if (!$this->twoFactorService->isEnabled($user)) {
            return back()->with('error', '2FA is not enabled for your account.');
        }

        $recoveryCodes = $this->twoFactorService->generateRecoveryCodes();
        $user->update(['two_factor_recovery_codes' => $recoveryCodes]);

        return view('auth.two-factor-recovery-codes', compact('recoveryCodes'));
    }
}