<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Auth\Events\Verified;
use Illuminate\Foundation\Auth\EmailVerificationRequest;

class VerificationController extends Controller
{
    /**
     * Show the email verification notice.
     */
    public function show()
    {
        $user = Auth::user();
        $otpService = app(\App\Services\OtpService::class);
        
        // If user must verify email with OTP
        if ($user->must_verify_email && !$user->hasVerifiedEmail()) {
            // Generate and send OTP automatically on first visit
            $otpService->generateAndSendOtp($user);
            
            $canResend = $otpService->canResend($user);
            $secondsUntilResend = $otpService->secondsUntilCanResend($user);
            $remainingAttempts = 5; // Spatie doesn't track attempts, we'll handle this differently
            
            return view('auth.verify-email', compact('user', 'canResend', 'secondsUntilResend', 'remainingAttempts'));
        }
        
        // For regular email verification (if needed)
        $canResend = true;
        $secondsUntilResend = 0;
        $remainingAttempts = 5;
        
        return $user->hasVerifiedEmail()
            ? redirect()->route('dashboard')
            : view('auth.verify-email', compact('user', 'canResend', 'secondsUntilResend', 'remainingAttempts'));
    }

    /**
     * Mark the authenticated user's email address as verified.
     */
    public function verify(EmailVerificationRequest $request)
    {
        $user = $request->user();
        
        // If user must verify with OTP, handle OTP verification
        if ($user->must_verify_email && !$user->hasVerifiedEmail()) {
            // This should be handled by POST route, redirect to verification page
            return redirect()->route('verification.notice');
        }
        
        if ($user->hasVerifiedEmail()) {
            return redirect()->route('dashboard')->with('verified', true);
        }

        if ($user->markEmailAsVerified()) {
            event(new Verified($user));
            
            // Log email verification using activity log
            activity()
                ->performedOn($user)
                ->causedBy($user)
                ->withProperties(['action' => 'email_verified'])
                ->log('Email address verified');
        }

        return redirect()->route('dashboard')->with('verified', true);
    }

    /**
     * Verify OTP code (POST method)
     */
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'otp_code' => ['required', 'string', 'size:6'],
        ]);

        $user = Auth::user();
        $otpService = app(\App\Services\OtpService::class);

        if ($otpService->validateOtp($user, $request->otp_code)) {
            // Update organization onboarding step
            if ($user->organization) {
                $user->organization->update([
                    'onboarding_step' => 'email_verified',
                ]);
            }

            // Log activity
            activity()
                ->performedOn($user)
                ->causedBy($user)
                ->withProperties(['action' => 'email_verified_otp'])
                ->log('Email address verified with OTP');

            return redirect()
                ->route('onboarding.index')
                ->with('success', 'Email verified successfully! Let\'s complete your setup.');
        }

        return back()->withErrors([
            'otp_code' => 'Invalid or expired OTP code. Please try again or request a new code.',
        ]);
    }

    /**
     * Resend the email verification notification.
     */
    public function resend(Request $request)
    {
        $user = $request->user();
        
        // If user must verify with OTP, resend OTP
        if ($user->must_verify_email && !$user->hasVerifiedEmail()) {
            $otpService = app(\App\Services\OtpService::class);
            
            if (!$otpService->canResend($user)) {
                $seconds = $otpService->secondsUntilCanResend($user);
                return back()->with('error', "Please wait {$seconds} seconds before requesting a new OTP.");
            }

            if ($otpService->generateAndSendOtp($user)) {
                return back()->with('success', 'A new OTP has been sent to your email.');
            }

            return back()->with('error', 'Failed to send OTP. Please try again.');
        }
        
        // Regular email verification resend (fallback)
        if ($user->hasVerifiedEmail()) {
            return redirect()->route('dashboard');
        }

        // This shouldn't happen with our flow, but keeping as fallback
        return back()->with('message', 'Email verification not required.');
    }
}