<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Department;
use App\Models\Announcement;
use App\Services\AnalyticsService;

class DashboardController extends Controller
{
    protected $analyticsService;

    public function __construct(AnalyticsService $analyticsService)
    {
        $this->analyticsService = $analyticsService;
    }

    public function index()
    {
        // Core KPIs
        $totalEmployees = Employee::where('status', 'active')->count();
        $totalDepartments = Department::where('active', true)->count();

        // Get analytics data from service
        $attendanceMetrics = $this->analyticsService->getTodayAttendanceMetrics();
        $leaveMetrics = $this->analyticsService->getLeaveMetrics();
        $payrollMetrics = $this->analyticsService->getPayrollMetrics();
        $performanceMetrics = $this->analyticsService->getPerformanceMetrics();

        // Recent employees
        $recentEmployees = Employee::with(['department', 'position'])
            ->where('status', 'active')
            ->latest()
            ->take(5)
            ->get();

        // Upcoming birthdays
        $upcomingBirthdays = Employee::whereMonth('date_of_birth', now()->month)
            ->whereDay('date_of_birth', '>=', now()->day)
            ->where('status', 'active')
            ->orderByRaw('DAY(date_of_birth)')
            ->take(5)
            ->get();

        // Active announcements
        $announcements = Announcement::active()
            ->published()
            ->orderBy('is_pinned', 'desc')
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        // Analytics data for charts
        $departmentAnalytics = $this->analyticsService->getDepartmentAnalytics();
        $attendanceTrends = $this->analyticsService->getAttendanceTrends();
        $leaveTrends = $this->analyticsService->getLeaveTrends();
        $payrollTrends = $this->analyticsService->getPayrollTrends();

        return view('dashboard', compact(
            'totalEmployees',
            'totalDepartments',
            'attendanceMetrics',
            'leaveMetrics',
            'payrollMetrics',
            'performanceMetrics',
            'recentEmployees',
            'upcomingBirthdays',
            'announcements',
            'departmentAnalytics',
            'attendanceTrends',
            'leaveTrends',
            'payrollTrends'
        ));
    }
}
