<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Exports\EmployeesExport;
use App\Imports\EmployeesImport;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Traits\SendsOrganizationMail;

class EmployeeController extends Controller
{
    use SendsOrganizationMail;
    public function index(Request $request)
    {
        $this->authorize('viewAny', Employee::class);

        // Optimized query with proper eager loading and tenant scoping
        $query = Employee::with([
            'department:id,name,active',
            'position:id,name,active', 
            'manager:id,first_name,last_name,employee_code'
        ]);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_code', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('department')) {
            $query->where('department_id', $request->department);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Group By functionality - optimized
        $groupBy = $request->input('group_by');
        $groupedEmployees = null;

        if ($groupBy && in_array($groupBy, ['department', 'status', 'position'])) {
            // Use select to limit data transfer
            $allEmployees = $query->select([
                'id', 'first_name', 'last_name', 'employee_code', 'email', 
                'status', 'department_id', 'position_id', 'organization_id'
            ])->get();

            if ($groupBy === 'department') {
                $groupedEmployees = $allEmployees->groupBy('department.name');
            } elseif ($groupBy === 'status') {
                $groupedEmployees = $allEmployees->groupBy('status');
            } elseif ($groupBy === 'position') {
                $groupedEmployees = $allEmployees->groupBy('position.name');
            }

            $employees = null;
        } else {
            $employees = $query->paginate(15);
        }

        // Optimize department and position queries with tenant scoping
        $departments = Department::select('id', 'name')->where('active', true)->get();
        $positions = Position::select('id', 'name')->where('active', true)->get();

        return view('employees.index', compact('employees', 'departments', 'positions', 'groupedEmployees', 'groupBy'));
    }

    public function create()
    {
        $this->authorize('create', Employee::class);

        // Optimized queries with select to reduce data transfer
        $departments = Department::select('id', 'name')->where('active', true)->get();
        $positions = Position::select('id', 'name')->where('active', true)->get();
        $managers = Employee::select('id', 'first_name', 'last_name', 'employee_code')
            ->where('status', 'active')
            ->get();

        return view('employees.create', compact('departments', 'positions', 'managers'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Employee::class);

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email',
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'nationality' => 'nullable|string|max:100',
            'identification_id' => 'nullable|string|max:50',
            'kra_pin' => 'nullable|string|max:20|unique:employees,kra_pin',
            'nhif_number' => 'nullable|string|max:20|unique:employees,nhif_number',
            'nssf_number' => 'nullable|string|max:20|unique:employees,nssf_number',
            'basic_salary' => 'nullable|numeric|min:0',
            'has_helb_loan' => 'boolean',
            'helb_monthly_deduction' => 'nullable|numeric|min:0',
            'department_id' => 'nullable|exists:departments,id',
            'position_id' => 'nullable|exists:positions,id',
            'manager_id' => 'nullable|exists:employees,id',
            'joining_date' => 'nullable|date',
            'work_location' => 'nullable|string|max:255',
            'work_email' => 'nullable|email',
            'work_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relation' => 'nullable|string|max:100',
            'bank_name' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:50',
            'bank_account_holder' => 'nullable|string|max:255',
            'photo' => 'nullable|image|max:2048',
        ]);

        // Generate random password
        $plainPassword = Str::random(12);

        // Create user account
        $user = User::create([
            'name' => $validated['first_name'] . ' ' . $validated['last_name'],
            'email' => $validated['email'],
            'password' => Hash::make($plainPassword),
            'organization_id' => Auth::user()->organization_id, // Set organization_id
        ]);

        // Assign default employee role
        $user->assignRole('employee');

        // Generate employee code
        $latestEmployee = Employee::latest('id')->first();
        $employeeCode = 'EMP' . str_pad(($latestEmployee ? $latestEmployee->id + 1 : 1), 5, '0', STR_PAD_LEFT);

        // Handle photo upload
        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('employee_photos', 'public');
        }

        $validated['user_id'] = $user->id;
        $validated['employee_code'] = $employeeCode;
        $validated['photo'] = $photoPath;
        $validated['status'] = 'active';

        $employee = Employee::create($validated);

        // Send welcome email with credentials using organization's email settings
        $emailSent = $this->sendOrganizationMail(
            $this->getCurrentOrganization(),
            $user->email,
            new \App\Mail\EmployeeAccountCreated($employee, $user, $plainPassword)
        );

        if ($emailSent) {
            return redirect()->route('employees.index')
                ->with('success', 'Employee created successfully! Welcome email sent to ' . $user->email);
        } else {
            return redirect()->route('employees.index')
                ->with('warning', 'Employee created successfully, but failed to send welcome email. Please check email configuration and logs.');
        }
    }

    public function show(Employee $employee)
    {
        $this->authorize('view', $employee);
        
        // Optimized eager loading with specific columns and limits
        $employee->load([
            'department:id,name',
            'position:id,name',
            'manager:id,first_name,last_name,employee_code',
            'subordinates:id,first_name,last_name,employee_code,manager_id',
            'contracts' => function ($query) {
                $query->select('id', 'employee_id', 'contract_type', 'start_date', 'end_date', 'salary', 'status')
                      ->latest()
                      ->take(5);
            },
            'attendances' => function ($query) {
                $query->select('id', 'employee_id', 'date', 'check_in', 'check_out', 'status')
                      ->latest()
                      ->take(10);
            },
            'leaveRequests' => function ($query) {
                $query->select('id', 'employee_id', 'leave_type_id', 'start_date', 'end_date', 'status', 'reason')
                      ->with('leaveType:id,name')
                      ->latest()
                      ->take(5);
            },
            'skills:id,name',
            'performanceReviews' => function ($query) {
                $query->select('id', 'employee_id', 'review_date', 'overall_rating', 'status')
                      ->latest()
                      ->take(5);
            },
            'documents' => function ($query) {
                $query->select('id', 'employee_id', 'type', 'file_name', 'created_at')
                      ->latest();
            }
        ]);

        return view('employees.show', compact('employee'));
    }

    public function edit(Employee $employee)
    {
        $this->authorize('update', $employee);

        // Optimized queries
        $departments = Department::select('id', 'name')->where('active', true)->get();
        $positions = Position::select('id', 'name')->where('active', true)->get();
        $managers = Employee::select('id', 'first_name', 'last_name', 'employee_code')
            ->where('status', 'active')
            ->where('id', '!=', $employee->id)
            ->get();

        return view('employees.edit', compact('employee', 'departments', 'positions', 'managers'));
    }

    public function update(Request $request, Employee $employee)
    {
        $this->authorize('update', $employee);

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email,' . $employee->id,
            'phone' => 'nullable|string|max:20',
            'mobile' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'nationality' => 'nullable|string|max:100',
            'identification_id' => 'nullable|string|max:50',
            'kra_pin' => 'nullable|string|max:20|unique:employees,kra_pin,' . $employee->id,
            'nhif_number' => 'nullable|string|max:20|unique:employees,nhif_number,' . $employee->id,
            'nssf_number' => 'nullable|string|max:20|unique:employees,nssf_number,' . $employee->id,
            'basic_salary' => 'nullable|numeric|min:0',
            'has_helb_loan' => 'boolean',
            'helb_monthly_deduction' => 'nullable|numeric|min:0',
            'department_id' => 'nullable|exists:departments,id',
            'position_id' => 'nullable|exists:positions,id',
            'manager_id' => 'nullable|exists:employees,id',
            'joining_date' => 'nullable|date',
            'work_location' => 'nullable|string|max:255',
            'work_email' => 'nullable|email',
            'work_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relation' => 'nullable|string|max:100',
            'bank_name' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:50',
            'bank_account_holder' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive,on_leave,terminated',
            'photo' => 'nullable|image|max:2048',
        ]);

        // Handle photo upload
        if ($request->hasFile('photo')) {
            if ($employee->photo) {
                Storage::disk('public')->delete($employee->photo);
            }
            $validated['photo'] = $request->file('photo')->store('employee_photos', 'public');
        }

        $employee->update($validated);

        // Update user
        $employee->user->update([
            'name' => $validated['first_name'] . ' ' . $validated['last_name'],
            'email' => $validated['email'],
        ]);

        return redirect()->route('employees.show', $employee)
            ->with('success', 'Employee updated successfully!');
    }

    public function destroy(Employee $employee)
    {
        $this->authorize('delete', $employee);

        if ($employee->photo) {
            Storage::disk('public')->delete($employee->photo);
        }

        $employee->delete();

        return redirect()->route('employees.index')
            ->with('success', 'Employee deleted successfully!');
    }

    public function export(Request $request)
    {
        $this->authorize('viewAny', Employee::class);
        
        $query = Employee::with(['department', 'position', 'manager']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_code', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('department')) {
            $query->where('department_id', $request->department);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $format = $request->input('format', 'xlsx');
        $filename = 'employees-' . now()->format('Y-m-d');

        if ($format === 'csv') {
            return Excel::download(new EmployeesExport($query), $filename . '.csv');
        }

        return Excel::download(new EmployeesExport($query), $filename . '.xlsx');
    }

    public function importView()
    {
        $this->authorize('create', Employee::class);

        return view('employees.import');
    }

    public function import(Request $request)
    {
        $this->authorize('create', Employee::class);

        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
        ]);

        try {
            Excel::import(new EmployeesImport, $request->file('file'));

            return redirect()->route('employees.index')
                ->with('success', 'Employees imported successfully!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error importing employees: ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $headers = [
            'employee_code',
            'first_name',
            'last_name',
            'email',
            'phone',
            'address',
            'city',
            'state',
            'zip_code',
            'date_of_birth',
            'gender',
            'marital_status',
            'nationality',
            'national_id',
            'kra_pin',
            'nhif_number',
            'nssf_number',
            'basic_salary',
            'department',
            'position',
            'join_date',
            'employment_type',
            'status',
            'emergency_contact',
            'emergency_phone',
            'bank_name',
            'bank_account',
            'password',
        ];

        return Excel::download(new class($headers) implements \Maatwebsite\Excel\Concerns\FromArray {
            private $headers;
            
            public function __construct($headers) {
                $this->headers = $headers;
            }
            
            public function array(): array {
                return [$this->headers];
            }
        }, 'employee-import-template.xlsx');
    }
}
