<?php

namespace App\Http\Controllers;

use App\Models\EmployeeLoan;
use App\Models\Employee;
use App\Models\LoanSettings;
use Illuminate\Http\Request;

class EmployeeLoanController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', EmployeeLoan::class);

        $query = EmployeeLoan::with(['employee', 'approver']);

        // Filter by status
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        // Filter by employee
        if ($request->has('employee_id') && $request->employee_id != '') {
            $query->where('employee_id', $request->employee_id);
        }

        // Filter by loan type
        if ($request->has('loan_type') && $request->loan_type != '') {
            $query->where('loan_type', $request->loan_type);
        }

        $loans = $query->orderBy('created_at', 'desc')->paginate(20);
        $employees = Employee::where('status', 'active')->get();

        return view('loans.index', compact('loans', 'employees'));
    }

    public function create()
    {
        $this->authorize('create', EmployeeLoan::class);

        $employees = Employee::where('status', 'active')->get();
        return view('loans.create', compact('employees'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', EmployeeLoan::class);
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:salary_advance,emergency,personal,education,housing,other',
            'principal_amount' => 'required|numeric|min:1',
            'interest_rate' => 'required|numeric|min:0|max:100',
            'repayment_period_months' => 'required|integer|min:1|max:60',
            'disbursement_date' => 'required|date',
            'first_deduction_date' => 'required|date|after:disbursement_date',
            'purpose' => 'nullable|string|max:500',
            'terms_conditions' => 'nullable|string',
        ]);

        $employee = Employee::findOrFail($validated['employee_id']);
        
        $loan = new EmployeeLoan($validated);
        $loan->organization_id = auth()->user()->organization_id;
        $loan->loan_number = EmployeeLoan::generateLoanNumber($employee);
        
        // Calculate totals
        $loan->total_amount = $loan->calculateTotalAmount();
        $loan->monthly_deduction = $loan->calculateMonthlyDeduction();
        $loan->balance = $loan->total_amount;
        
        // Calculate expected completion date
        $loan->expected_completion_date = \Carbon\Carbon::parse($loan->first_deduction_date)
            ->addMonths($loan->repayment_period_months);
        
        $loan->save();

        return redirect()->route('loans.show', $loan)
            ->with('success', 'Loan created successfully. Awaiting approval.');
    }

    public function show(EmployeeLoan $loan)
    {
        $this->authorize('view', $loan);

        $loan->load(['employee', 'approver', 'repayments.payroll']);
        return view('loans.show', compact('loan'));
    }

    public function edit(EmployeeLoan $loan)
    {
        $this->authorize('update', $loan);

        $employees = Employee::where('status', 'active')->get();
        return view('loans.edit', compact('loan', 'employees'));
    }

    public function update(Request $request, EmployeeLoan $loan)
    {
        $this->authorize('update', $loan);

        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'loan_type' => 'required|in:salary_advance,emergency,personal,education,housing,other',
            'principal_amount' => 'required|numeric|min:1',
            'interest_rate' => 'required|numeric|min:0|max:100',
            'repayment_period_months' => 'required|integer|min:1|max:60',
            'disbursement_date' => 'required|date',
            'first_deduction_date' => 'required|date|after:disbursement_date',
            'purpose' => 'nullable|string|max:500',
            'terms_conditions' => 'nullable|string',
        ]);

        $loan->fill($validated);
        
        // Recalculate totals
        $loan->total_amount = $loan->calculateTotalAmount();
        $loan->monthly_deduction = $loan->calculateMonthlyDeduction();
        $loan->balance = $loan->total_amount;
        
        $loan->expected_completion_date = \Carbon\Carbon::parse($loan->first_deduction_date)
            ->addMonths($loan->repayment_period_months);
        
        $loan->save();

        return redirect()->route('loans.show', $loan)
            ->with('success', 'Loan updated successfully.');
    }

    public function approve(Request $request, EmployeeLoan $loan)
    {
        $this->authorize('approve', $loan);

        $request->validate([
            'approval_notes' => 'nullable|string|max:500',
        ]);

        $loan->status = 'approved';
        $loan->approved_by = auth()->id();
        $loan->approved_at = now();
        $loan->approval_notes = $request->approval_notes;
        $loan->save();

        return back()->with('success', 'Loan approved successfully.');
    }

    public function disburse(EmployeeLoan $loan)
    {
        $this->authorize('disburse', $loan);

        $loan->status = 'disbursed';
        $loan->save();

        return back()->with('success', 'Loan marked as disbursed.');
    }

    public function activate(EmployeeLoan $loan)
    {
        $this->authorize('activate', $loan);

        $loan->status = 'active';
        $loan->save();

        return back()->with('success', 'Loan activated. Deductions will start from next payroll.');
    }

    public function reject(Request $request, EmployeeLoan $loan)
    {
        $this->authorize('reject', $loan);

        $request->validate([
            'approval_notes' => 'required|string|max:500',
        ]);

        $loan->status = 'cancelled';
        $loan->approved_by = auth()->id();
        $loan->approved_at = now();
        $loan->approval_notes = $request->approval_notes;
        $loan->save();

        return back()->with('success', 'Loan rejected.');
    }

    public function destroy(EmployeeLoan $loan)
    {
        $this->authorize('delete', $loan);

        $loan->delete();

        return redirect()->route('loans.index')
            ->with('success', 'Loan deleted successfully.');
    }

    /**
     * Show employee's own loans
     */
    public function myLoans()
    {
        $employee = auth()->user()->employee;
        
        if (!$employee) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Only employees can access this page.']);
        }

        // Check if user has organization
        if (!auth()->user()->organization_id) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'You must belong to an organization to access loans.']);
        }

        $loans = $employee->loans()
            ->with(['approver', 'repayments'])
            ->orderBy('created_at', 'desc')
            ->get();

        $loanSettings = LoanSettings::getForOrganization(auth()->user()->organization_id);

        return view('loans.my-loans', compact('loans', 'loanSettings'));
    }

    /**
     * Show loan request form for employees
     */
    public function requestForm()
    {
        $employee = auth()->user()->employee;
        
        if (!$employee) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Only employees can request loans.']);
        }

        // Check if user has organization
        if (!auth()->user()->organization_id) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'You must belong to an organization to request loans.']);
        }

        $loanSettings = LoanSettings::getForOrganization(auth()->user()->organization_id);

        return view('loans.request', compact('employee', 'loanSettings'));
    }

    /**
     * Submit loan request from employee
     */
    public function submitRequest(Request $request)
    {
        $employee = auth()->user()->employee;
        
        if (!$employee) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Only employees can request loans.']);
        }

        // Check if user has organization
        if (!auth()->user()->organization_id) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'You must belong to an organization to request loans.']);
        }

        $loanSettings = LoanSettings::getForOrganization(auth()->user()->organization_id);

        $validated = $request->validate([
            'loan_type' => 'required|in:salary_advance,emergency,personal,education,housing,other',
            'principal_amount' => 'required|numeric|min:1',
            'repayment_period_months' => 'required|integer|min:1|max:' . $loanSettings->max_loan_term_months,
            'purpose' => 'required|string|max:500',
        ]);

        // Check eligibility
        $eligibility = $loanSettings->isEmployeeEligible($employee, $validated['principal_amount']);
        
        if (!$eligibility['eligible']) {
            return back()
                ->withInput()
                ->withErrors(['eligibility' => $eligibility['errors']]);
        }

        // Get default interest rate for loan type
        $interestRate = $loanSettings->getInterestRateForType($validated['loan_type']);

        $loan = new EmployeeLoan($validated);
        $loan->employee_id = $employee->id;
        $loan->organization_id = auth()->user()->organization_id;
        $loan->loan_number = EmployeeLoan::generateLoanNumber($employee);
        $loan->interest_rate = $interestRate;
        $loan->disbursement_date = now()->addDays(7); // 7 days from now
        $loan->first_deduction_date = now()->addMonth(); // Next month
        
        // Calculate totals
        $loan->total_amount = $loan->calculateTotalAmount();
        $loan->monthly_deduction = $loan->calculateMonthlyDeduction();
        $loan->balance = $loan->total_amount;
        
        // Calculate expected completion date
        $loan->expected_completion_date = \Carbon\Carbon::parse($loan->first_deduction_date)
            ->addMonths((int) $loan->repayment_period_months);
        
        $loan->status = 'pending'; // Awaiting approval
        $loan->save();

        return redirect()->route('loans.my-loans')
            ->with('success', 'Loan request submitted successfully. Awaiting approval.');
    }
}
