<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Interview;
use App\Models\Application;
use Illuminate\Support\Str;

class InterviewController extends Controller
{
    /**
     * Store a newly created interview
     */
    public function store(Request $request)
    {
        // Check authentication and permissions
        if (!auth()->check()) {
            return redirect()->route('login');
        }

        $validated = $request->validate([
            'application_id' => 'required|exists:applications,id',
            'scheduled_at' => 'required|date|after:now',
            'location' => 'nullable|string|max:255',
            'meeting_link' => 'nullable|url|max:500',
            'interviewer_id' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        // Get the application and verify tenant access
        $application = Application::findOrFail($validated['application_id']);
        
        if ($application->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied');
        }

        // Create the interview
        $interview = Interview::create([
            'application_id' => $application->id,
            'scheduled_at' => $validated['scheduled_at'],
            'location' => $validated['location'] ?? null,
            'meeting_link' => $validated['meeting_link'] ?? null,
            'interviewer_id' => $validated['interviewer_id'] ?? auth()->id(),
            'notes' => $validated['notes'] ?? null,
            'status' => 'scheduled',
            'confirmation_token' => Str::random(32),
            'organization_id' => auth()->user()->organization_id,
        ]);

        // Update application status to interviewed
        $application->update(['status' => 'interviewed']);

        // Send interview invitation email
        try {
            \Mail::to($application->email)->send(
                new \App\Mail\InterviewScheduled($interview, $application)
            );
        } catch (\Exception $e) {
            \Log::error('Failed to send interview invitation email: ' . $e->getMessage());
        }

        return redirect()
            ->back()
            ->with('success', 'Interview scheduled successfully and invitation sent to candidate.');
    }

    /**
     * Display the specified interview
     */
    public function show(Interview $interview)
    {
        // Check tenant access
        if ($interview->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied');
        }

        $interview->load(['application', 'interviewer', 'questions']);

        return view('recruitment.interviews.show', compact('interview'));
    }

    /**
     * Update the specified interview
     */
    public function update(Request $request, Interview $interview)
    {
        // Check tenant access
        if ($interview->organization_id !== auth()->user()->organization_id) {
            abort(403, 'Access denied');
        }

        $validated = $request->validate([
            'scheduled_at' => 'required|date',
            'location' => 'nullable|string|max:255',
            'meeting_link' => 'nullable|url|max:500',
            'interviewer_id' => 'nullable|exists:users,id',
            'status' => 'nullable|in:scheduled,completed,cancelled,rescheduled',
            'notes' => 'nullable|string',
        ]);

        $interview->update($validated);

        return redirect()
            ->back()
            ->with('success', 'Interview updated successfully.');
    }
}
