<?php

namespace App\Http\Controllers;

use App\Models\JobPosting;
use App\Models\Department;
use App\Models\Position;
use App\Models\Application;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class JobPostingController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_job_postings')->only(['index', 'show']);
        $this->middleware('permission:create_job_postings')->only(['create', 'store']);
        $this->middleware('permission:edit_job_postings')->only(['edit', 'update']);
        $this->middleware('permission:delete_job_postings')->only(['destroy']);
    }

    /**
     * Display a listing of job postings
     */
    public function index(Request $request)
    {
        $query = JobPosting::with(['department:id,name', 'position:id,name', 'creator:id,name'])
                          ->withCount('applications');

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('location', 'like', "%{$search}%")
                  ->orWhereHas('department', function ($dept) use ($search) {
                      $dept->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by department
        if ($request->filled('department')) {
            $query->where('department_id', $request->department);
        }

        $jobPostings = $query->latest()->paginate(15);

        // Get statistics for dashboard cards
        $stats = [
            'open_positions' => JobPosting::active()->count(),
            'total_applications' => Application::count(),
            'interviews_scheduled' => \App\Models\Interview::where('status', 'scheduled')->count(),
            'offers_extended' => Application::where('status', 'offered')->count(),
        ];

        $departments = Department::select('id', 'name')->where('active', true)->get();

        return view('recruitment.job-postings.index', compact('jobPostings', 'stats', 'departments'));
    }

    /**
     * Show the form for creating a new job posting
     */
    public function create()
    {
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();

        return view('recruitment.job-postings.create', compact('departments', 'positions'));
    }

    /**
     * Store a newly created job posting
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'department_id' => 'required|exists:departments,id',
            'position_id' => 'nullable|exists:positions,id',
            'location' => 'required|string|max:255',
            'employment_type' => 'required|in:full-time,part-time,contract,internship',
            'description' => 'required|string',
            'requirements' => 'required|string',
            'responsibilities' => 'required|string',
            'salary_range' => 'nullable|string|max:100',
            'closes_at' => 'nullable|date|after:today',
            'status' => 'required|in:draft,active',
        ]);

        $validated['created_by'] = Auth::id();

        $jobPosting = JobPosting::create($validated);

        return redirect()->route('job-postings.index')
                        ->with('success', 'Job posting created successfully!');
    }

    /**
     * Display the specified job posting
     */
    public function show(JobPosting $jobPosting)
    {
        $jobPosting->load([
            'department:id,name',
            'position:id,name',
            'creator:id,name',
            'applications' => function ($query) {
                $query->with('statusHistory.changedBy:id,name')
                      ->latest('applied_at');
            }
        ]);

        $applicationStats = [
            'total' => $jobPosting->applications->count(),
            'new' => $jobPosting->applications->where('status', 'new')->count(),
            'screening' => $jobPosting->applications->where('status', 'screening')->count(),
            'shortlisted' => $jobPosting->applications->where('status', 'shortlisted')->count(),
            'interviewed' => $jobPosting->applications->where('status', 'interviewed')->count(),
            'offered' => $jobPosting->applications->where('status', 'offered')->count(),
            'rejected' => $jobPosting->applications->where('status', 'rejected')->count(),
        ];

        return view('recruitment.job-postings.show', compact('jobPosting', 'applicationStats'));
    }

    /**
     * Show the form for editing the specified job posting
     */
    public function edit(JobPosting $jobPosting)
    {
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();

        return view('recruitment.job-postings.edit', compact('jobPosting', 'departments', 'positions'));
    }

    /**
     * Update the specified job posting
     */
    public function update(Request $request, JobPosting $jobPosting)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'department_id' => 'required|exists:departments,id',
            'position_id' => 'nullable|exists:positions,id',
            'location' => 'required|string|max:255',
            'employment_type' => 'required|in:full-time,part-time,contract,internship',
            'description' => 'required|string',
            'requirements' => 'required|string',
            'responsibilities' => 'required|string',
            'salary_range' => 'nullable|string|max:100',
            'closes_at' => 'nullable|date|after:today',
            'status' => 'required|in:draft,active,closed',
        ]);

        $jobPosting->update($validated);

        return redirect()->route('job-postings.show', $jobPosting)
                        ->with('success', 'Job posting updated successfully!');
    }

    /**
     * Remove the specified job posting
     */
    public function destroy(JobPosting $jobPosting)
    {
        // Check if there are applications
        if ($jobPosting->applications()->count() > 0) {
            return redirect()->route('job-postings.index')
                           ->with('error', 'Cannot delete job posting with existing applications.');
        }

        $jobPosting->delete();

        return redirect()->route('job-postings.index')
                        ->with('success', 'Job posting deleted successfully!');
    }

    /**
     * Close a job posting
     */
    public function close(JobPosting $jobPosting)
    {
        $this->authorize('edit_job_postings');

        $jobPosting->update(['status' => 'closed']);

        return redirect()->back()
                        ->with('success', 'Job posting closed successfully!');
    }
}