<?php

namespace App\Http\Controllers;

use App\Models\LearningPath;
use App\Models\LearningPathStep;
use App\Models\LearningPathEnrollment;
use App\Models\TrainingProgram;
use App\Models\TrainingEnrollment;
use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class LearningPathController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', LearningPath::class);

        $query = LearningPath::with(['creator', 'steps.trainingProgram']);

        // Filter by difficulty
        if ($request->filled('difficulty')) {
            $query->where('difficulty_level', $request->difficulty);
        }

        // Filter by active status
        if ($request->filled('status')) {
            $isActive = $request->status === 'active';
            $query->where('is_active', $isActive);
        } else {
            // Default to active paths only for regular users
            if (!auth()->user()->can('view_inactive_learning_paths')) {
                $query->where('is_active', true);
            }
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $learningPaths = $query->orderBy('title')->paginate(12);

        // Get user's enrollments
        $userEnrollments = [];
        if (auth()->user()->employee) {
            $userEnrollments = LearningPathEnrollment::where('employee_id', auth()->user()->employee->id)
                ->pluck('learning_path_id')
                ->toArray();
        }

        return view('training.learning-paths.index', compact('learningPaths', 'userEnrollments'));
    }

    public function create()
    {
        $this->authorize('create', LearningPath::class);

        $trainingPrograms = TrainingProgram::where('is_active', true)->with('category')->get();
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();

        return view('training.learning-paths.create', compact('trainingPrograms', 'departments', 'positions'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', LearningPath::class);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'estimated_duration_hours' => 'required|integer|min:1',
            'target_roles' => 'nullable|array',
            'learning_outcomes' => 'nullable|array',
            'learning_outcomes.*' => 'string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'steps' => 'required|array|min:1',
            'steps.*.training_program_id' => 'required|exists:training_programs,id',
            'steps.*.is_required' => 'boolean',
            'steps.*.prerequisites' => 'nullable|array',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $validated['image_path'] = $request->file('image')->store('learning-paths', 'public');
        }

        $validated['created_by'] = auth()->id();
        $validated['organization_id'] = auth()->user()->organization_id;

        $learningPath = LearningPath::create($validated);

        // Create steps
        foreach ($validated['steps'] as $index => $stepData) {
            LearningPathStep::create([
                'learning_path_id' => $learningPath->id,
                'training_program_id' => $stepData['training_program_id'],
                'step_order' => $index + 1,
                'is_required' => $stepData['is_required'] ?? true,
                'prerequisites' => $stepData['prerequisites'] ?? [],
            ]);
        }

        return redirect()->route('training.learning-paths.show', $learningPath)
            ->with('success', 'Learning path created successfully!');
    }

    public function show(LearningPath $learningPath)
    {
        $this->authorize('view', $learningPath);

        $learningPath->load([
            'creator',
            'steps.trainingProgram.category',
            'enrollments' => function($query) {
                $query->with('employee')->latest();
            }
        ]);

        // Get user's enrollment if exists
        $userEnrollment = null;
        if (auth()->user()->employee) {
            $userEnrollment = LearningPathEnrollment::where('learning_path_id', $learningPath->id)
                ->where('employee_id', auth()->user()->employee->id)
                ->first();
        }

        // Statistics
        $stats = [
            'total_enrollments' => $learningPath->enrollments()->count(),
            'active_enrollments' => $learningPath->enrollments()->active()->count(),
            'completed_enrollments' => $learningPath->enrollments()->completed()->count(),
            'completion_rate' => $learningPath->completion_rate,
        ];

        return view('training.learning-paths.show', compact('learningPath', 'userEnrollment', 'stats'));
    }

    public function edit(LearningPath $learningPath)
    {
        $this->authorize('update', $learningPath);

        $learningPath->load('steps.trainingProgram');
        $trainingPrograms = TrainingProgram::where('is_active', true)->with('category')->get();
        $departments = Department::where('active', true)->get();
        $positions = Position::where('active', true)->get();

        return view('training.learning-paths.edit', compact('learningPath', 'trainingPrograms', 'departments', 'positions'));
    }

    public function update(Request $request, LearningPath $learningPath)
    {
        $this->authorize('update', $learningPath);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'estimated_duration_hours' => 'required|integer|min:1',
            'target_roles' => 'nullable|array',
            'learning_outcomes' => 'nullable|array',
            'learning_outcomes.*' => 'string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'steps' => 'required|array|min:1',
            'steps.*.training_program_id' => 'required|exists:training_programs,id',
            'steps.*.is_required' => 'boolean',
            'steps.*.prerequisites' => 'nullable|array',
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($learningPath->image_path) {
                Storage::disk('public')->delete($learningPath->image_path);
            }
            $validated['image_path'] = $request->file('image')->store('learning-paths', 'public');
        }

        $learningPath->update($validated);

        // Update steps - delete existing and recreate
        $learningPath->steps()->delete();
        
        foreach ($validated['steps'] as $index => $stepData) {
            LearningPathStep::create([
                'learning_path_id' => $learningPath->id,
                'training_program_id' => $stepData['training_program_id'],
                'step_order' => $index + 1,
                'is_required' => $stepData['is_required'] ?? true,
                'prerequisites' => $stepData['prerequisites'] ?? [],
            ]);
        }

        return redirect()->route('training.learning-paths.show', $learningPath)
            ->with('success', 'Learning path updated successfully!');
    }

    public function destroy(LearningPath $learningPath)
    {
        $this->authorize('delete', $learningPath);

        // Check if there are active enrollments
        $activeEnrollments = $learningPath->enrollments()->active()->count();
        if ($activeEnrollments > 0) {
            return back()->with('error', 'Cannot delete learning path with active enrollments.');
        }

        // Delete image if exists
        if ($learningPath->image_path) {
            Storage::disk('public')->delete($learningPath->image_path);
        }

        $learningPath->delete();

        return redirect()->route('training.learning-paths.index')
            ->with('success', 'Learning path deleted successfully!');
    }

    public function enroll(Request $request, LearningPath $learningPath)
    {
        $this->authorize('enroll', $learningPath);

        $validated = $request->validate([
            'employee_id' => 'nullable|exists:employees,id',
        ]);

        // Default to current user's employee record
        $employeeId = $validated['employee_id'] ?? auth()->user()->employee?->id;

        if (!$employeeId) {
            return back()->with('error', 'Employee record not found.');
        }

        // Check if already enrolled
        $existingEnrollment = LearningPathEnrollment::where('learning_path_id', $learningPath->id)
            ->where('employee_id', $employeeId)
            ->first();

        if ($existingEnrollment) {
            return back()->with('error', 'Already enrolled in this learning path.');
        }

        LearningPathEnrollment::create([
            'learning_path_id' => $learningPath->id,
            'employee_id' => $employeeId,
            'status' => 'enrolled',
            'enrolled_at' => now(),
            'current_step' => 1,
            'enrolled_by' => auth()->id(),
        ]);

        return back()->with('success', 'Successfully enrolled in learning path!');
    }

    public function myPaths()
    {
        $employee = auth()->user()->employee;
        if (!$employee) {
            return redirect()->route('training.learning-paths.index')
                ->with('error', 'Employee record not found.');
        }

        $enrollments = LearningPathEnrollment::where('employee_id', $employee->id)
            ->with(['learningPath.steps.trainingProgram'])
            ->orderBy('enrolled_at', 'desc')
            ->paginate(10);

        return view('training.learning-paths.my-paths', compact('enrollments'));
    }

    public function pathProgress(LearningPathEnrollment $enrollment)
    {
        // Check if user can view this enrollment
        if (auth()->user()->employee?->id !== $enrollment->employee_id && 
            !auth()->user()->can('view_all_learning_path_progress')) {
            abort(403);
        }

        $enrollment->load([
            'learningPath.steps.trainingProgram.category',
            'employee'
        ]);

        // Get completion status for each step
        $stepProgress = [];
        foreach ($enrollment->learningPath->steps as $step) {
            $isCompleted = TrainingEnrollment::where('employee_id', $enrollment->employee_id)
                ->where('status', 'completed')
                ->whereHas('session', function($query) use ($step) {
                    $query->where('training_program_id', $step->training_program_id);
                })
                ->exists();

            $stepProgress[$step->id] = [
                'completed' => $isCompleted,
                'step' => $step,
            ];
        }

        return view('training.learning-paths.progress', compact('enrollment', 'stepProgress'));
    }

    public function updateProgress(LearningPathEnrollment $enrollment)
    {
        // Check permissions
        if (auth()->user()->employee?->id !== $enrollment->employee_id && 
            !auth()->user()->can('manage_learning_path_enrollments')) {
            abort(403);
        }

        $enrollment->updateProgress();

        return back()->with('success', 'Progress updated successfully!');
    }
}