<?php

namespace App\Http\Controllers;

use App\Models\LoanSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoanSettingsController extends Controller
{
    /**
     * Display loan settings
     */
    public function index()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific loan settings.');
        }
        
        $settings = LoanSettings::getForOrganization($user->organization_id);
        
        return view('loan-settings.index', compact('settings'));
    }

    /**
     * Show the form for editing loan settings
     */
    public function edit()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific loan settings.');
        }
        
        $settings = LoanSettings::getForOrganization($user->organization_id);
        
        return view('loan-settings.edit', compact('settings'));
    }

    /**
     * Update loan settings
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific loan settings.');
        }
        
        $settings = LoanSettings::getForOrganization($user->organization_id);

        $validated = $request->validate([
            'max_loan_amount' => 'required|numeric|min:1',
            'max_loan_term_months' => 'required|integer|min:1|max:120',
            'min_employment_tenure_months' => 'required|integer|min:0|max:60',
            'max_outstanding_loans' => 'required|integer|min:1|max:10',
            'loan_to_salary_ratio' => 'required|numeric|min:0.1|max:10',
            'auto_approve_below_amount' => 'required|numeric|min:0',
            'require_guarantor_above_amount' => 'required|numeric|min:0',
            'ceo_approval_threshold' => 'required|numeric|min:0',
            'require_department_head_approval' => 'boolean',
            'require_finance_approval' => 'boolean',
            'require_ceo_approval_above_threshold' => 'boolean',
            'allow_multiple_loans' => 'boolean',
            'allow_loan_with_pending_loan' => 'boolean',
            'allow_early_repayment' => 'boolean',
            'allow_payment_holidays' => 'boolean',
            'max_payment_holidays' => 'required|integer|min:0|max:12',
            'early_repayment_penalty_rate' => 'required|numeric|min:0|max:100',
            'notify_on_approval' => 'boolean',
            'notify_before_deduction' => 'boolean',
            'notification_days_before' => 'required|integer|min:1|max:30',
            'terms_and_conditions' => 'nullable|string',
            
            // Interest rates
            'interest_salary_advance' => 'required|numeric|min:0|max:100',
            'interest_emergency' => 'required|numeric|min:0|max:100',
            'interest_personal' => 'required|numeric|min:0|max:100',
            'interest_education' => 'required|numeric|min:0|max:100',
            'interest_housing' => 'required|numeric|min:0|max:100',
            'interest_other' => 'required|numeric|min:0|max:100',
        ]);

        // Build interest rates array
        $interestRates = [
            'salary_advance' => $validated['interest_salary_advance'],
            'emergency' => $validated['interest_emergency'],
            'personal' => $validated['interest_personal'],
            'education' => $validated['interest_education'],
            'housing' => $validated['interest_housing'],
            'other' => $validated['interest_other'],
        ];

        // Remove interest rate fields from validated data
        unset($validated['interest_salary_advance'], $validated['interest_emergency'], 
              $validated['interest_personal'], $validated['interest_education'], 
              $validated['interest_housing'], $validated['interest_other']);

        // Add interest rates to validated data
        $validated['default_interest_rates'] = $interestRates;

        $settings->update($validated);

        return redirect()->route('loan-settings.index')
            ->with('success', 'Loan settings updated successfully.');
    }

    /**
     * Reset loan settings to defaults
     */
    public function reset()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific loan settings.');
        }
        
        $settings = LoanSettings::getForOrganization($user->organization_id);

        $settings->update([
            'max_loan_amount' => 500000,
            'max_loan_term_months' => 36,
            'min_employment_tenure_months' => 6,
            'max_outstanding_loans' => 2,
            'loan_to_salary_ratio' => 3.0,
            'default_interest_rates' => LoanSettings::getDefaultInterestRates(),
            'auto_approve_below_amount' => 10000,
            'require_guarantor_above_amount' => 100000,
            'require_department_head_approval' => true,
            'require_finance_approval' => true,
            'require_ceo_approval_above_threshold' => true,
            'ceo_approval_threshold' => 200000,
            'allow_multiple_loans' => true,
            'allow_loan_with_pending_loan' => false,
            'allow_early_repayment' => true,
            'allow_payment_holidays' => false,
            'max_payment_holidays' => 2,
            'early_repayment_penalty_rate' => 0,
            'notify_on_approval' => true,
            'notify_before_deduction' => true,
            'notification_days_before' => 3,
        ]);

        return redirect()->route('loan-settings.index')
            ->with('success', 'Loan settings reset to defaults.');
    }
}
