<?php

namespace App\Http\Controllers;

use App\Models\SubscriptionPlan;
use App\Services\SubscriptionService;
use App\Services\BillingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class OnboardingController extends Controller
{
    protected $subscriptionService;
    protected $billingService;

    public function __construct(
        SubscriptionService $subscriptionService,
        BillingService $billingService
    ) {
        $this->middleware(['auth', 'verified']);
        $this->subscriptionService = $subscriptionService;
        $this->billingService = $billingService;
    }

    /**
     * Show current onboarding step
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function index()
    {
        $user = Auth::user();
        $organization = $user->organization;

        if (!$organization) {
            return redirect()->route('dashboard');
        }

        // Route to appropriate step based on onboarding status
        switch ($organization->onboarding_step) {
            case 'pending_verification':
                return redirect()->route('verification.notice');
                
            case 'email_verified':
                return redirect()->route('onboarding.welcome');
                
            case 'plan_selection':
                return redirect()->route('onboarding.select-plan');
                
            case 'payment_pending':
                return redirect()->route('onboarding.payment');
                
            case 'completed':
                return redirect()->route('dashboard');
                
            default:
                return redirect()->route('onboarding.welcome');
        }
    }

    /**
     * Show welcome step
     *
     * @return \Illuminate\View\View
     */
    public function welcome()
    {
        $user = Auth::user();
        $organization = $user->organization;

        return view('onboarding.welcome', compact('user', 'organization'));
    }

    /**
     * Show plan selection step
     *
     * @return \Illuminate\View\View
     */
    public function selectPlan()
    {
        $plans = SubscriptionPlan::active()->ordered()->get();
        $organization = Auth::user()->organization;

        return view('onboarding.select-plan', compact('plans', 'organization'));
    }

    /**
     * Store selected plan
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storePlan(Request $request)
    {
        $request->validate([
            'plan_id' => ['required', 'exists:subscription_plans,id'],
            'billing_cycle' => ['required', 'in:monthly,annual'],
            'start_with_trial' => ['boolean'],
        ]);

        $user = Auth::user();
        $organization = $user->organization;
        $plan = SubscriptionPlan::findOrFail($request->plan_id);
        $startWithTrial = $request->boolean('start_with_trial', true);

        try {
            // Create subscription
            $subscription = $this->subscriptionService->createSubscription(
                $organization,
                $plan,
                $request->billing_cycle,
                $startWithTrial
            );

            Log::info('Subscription created during onboarding', [
                'organization_id' => $organization->id,
                'plan_id' => $plan->id,
                'with_trial' => $startWithTrial,
            ]);

            // Log activity
            activity()
                ->performedOn($organization)
                ->causedBy($user)
                ->withProperties([
                    'plan' => $plan->name,
                    'billing_cycle' => $request->billing_cycle,
                    'with_trial' => $startWithTrial,
                ])
                ->log('Subscription plan selected during onboarding');

            // If trial, complete onboarding immediately
            if ($startWithTrial) {
                $organization->update([
                    'onboarding_step' => 'completed',
                    'onboarding_completed_at' => now(),
                ]);

                $user->update([
                    'first_login_completed' => true,
                ]);

                // Log onboarding completion
                activity()
                    ->performedOn($organization)
                    ->causedBy($user)
                    ->withProperties(['plan' => $plan->name, 'trial' => true])
                    ->log('Onboarding completed with trial subscription');

                // Send welcome email
                // Mail::to($user->email)->send(new TrialStarted($organization, $subscription));

                return redirect()
                    ->route('onboarding.complete')
                    ->with('success', 'Your 14-day trial has started!');
            }

            // If paid plan, go to payment
            $organization->update([
                'onboarding_step' => 'payment_pending',
            ]);

            return redirect()->route('onboarding.payment');

        } catch (\Exception $e) {
            Log::error('Failed to create subscription during onboarding', [
                'organization_id' => $organization->id,
                'error' => $e->getMessage(),
            ]);

            return back()
                ->withInput()
                ->with('error', 'Failed to create subscription. Please try again.');
        }
    }

    /**
     * Show payment step
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function payment()
    {
        $organization = Auth::user()->organization;
        $subscription = $organization->subscription;

        if (!$subscription) {
            return redirect()->route('onboarding.select-plan');
        }

        // Generate invoice if not exists
        $invoice = $subscription->invoices()->where('status', 'pending')->first();
        
        if (!$invoice) {
            $invoice = $this->billingService->generateSubscriptionInvoice($subscription);
        }

        return view('onboarding.payment', compact('organization', 'subscription', 'invoice'));
    }

    /**
     * Show completion step
     *
     * @return \Illuminate\View\View
     */
    public function complete()
    {
        $user = Auth::user();
        $organization = $user->organization;
        $subscription = $organization->subscription;

        return view('onboarding.complete', compact('user', 'organization', 'subscription'));
    }

    /**
     * Mark onboarding as complete and redirect to dashboard
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function finish()
    {
        $user = Auth::user();
        $organization = $user->organization;

        $organization->update([
            'onboarding_step' => 'completed',
            'onboarding_completed_at' => now(),
        ]);

        $user->update([
            'first_login_completed' => true,
        ]);

        return redirect()
            ->route('dashboard')
            ->with('success', 'Welcome to Qaya HRM! Let\'s get started.');
    }
}
