<?php

namespace App\Http\Controllers\Organization;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\User;
use App\Services\AuditService;
use App\Exports\AuditLogsExport;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class AuditLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_organization_audit_logs');
    }

    /**
     * Organization Audit Dashboard - can only see their organization's logs
     */
    public function index(Request $request): View|JsonResponse
    {
        $query = AuditLog::with(['user', 'organization'])
            ->forOrganization(Auth::user()->organization_id)
            ->latest();

        // Apply filters
        $this->applyFilters($query, $request);

        // Handle AJAX requests for real-time filtering
        if ($request->ajax()) {
            $logs = $query->paginate(25);
            return response()->json([
                'html' => view('organization.audit-logs.table', compact('logs'))->render(),
                'pagination' => $logs->links()->render(),
                'total' => $logs->total(),
            ]);
        }

        $logs = $query->paginate(25);
        
        // Get statistics for this organization only
        $statistics = AuditService::getStatistics(Auth::user()->organization_id, 30);
        
        // Get organization users for filter dropdown
        $organizationUsers = User::where('organization_id', Auth::user()->organization_id)
            ->select('id', 'name', 'email')
            ->get();

        return view('organization.audit-logs.index', compact('logs', 'statistics', 'organizationUsers'));
    }

    /**
     * Organization Dashboard - Overview of organization audit activities
     */
    public function dashboard(Request $request): View
    {
        $days = $request->get('days', 30);
        $organizationId = Auth::user()->organization_id;
        
        // Get organization statistics
        $statistics = AuditService::getStatistics($organizationId, $days);
        
        // Get department-wise activity (if departments exist)
        $departmentStats = [];
        if (class_exists('App\Models\Department')) {
            $departments = \App\Models\Department::where('organization_id', $organizationId)->get();
            foreach ($departments as $dept) {
                $deptUserIds = User::where('organization_id', $organizationId)
                    ->whereHas('employee', function($q) use ($dept) {
                        $q->where('department_id', $dept->id);
                    })
                    ->pluck('id');
                
                $deptActivityCount = AuditLog::whereIn('user_id', $deptUserIds)
                    ->where('created_at', '>=', now()->subDays($days))
                    ->count();
                    
                $departmentStats[] = [
                    'department' => $dept,
                    'activity_count' => $deptActivityCount
                ];
            }
        }

        // Get recent critical activities
        $criticalActivities = AuditLog::with(['user'])
            ->forOrganization($organizationId)
            ->whereIn('severity', ['high', 'critical'])
            ->where('created_at', '>=', now()->subDays($days))
            ->latest()
            ->limit(10)
            ->get();

        // Get top active users in organization
        $topUsers = AuditLog::selectRaw('user_id, COUNT(*) as activity_count')
            ->forOrganization($organizationId)
            ->whereNotNull('user_id')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('user_id')
            ->orderByDesc('activity_count')
            ->limit(5)
            ->with('user')
            ->get();

        // Get compliance-related activities
        $complianceActivities = AuditLog::with(['user'])
            ->forOrganization($organizationId)
            ->where('category', 'compliance')
            ->where('created_at', '>=', now()->subDays($days))
            ->latest()
            ->limit(5)
            ->get();

        return view('organization.audit-logs.dashboard', compact(
            'statistics', 
            'departmentStats', 
            'criticalActivities', 
            'topUsers',
            'complianceActivities',
            'days'
        ));
    }

    /**
     * Show detailed audit log entry
     */
    public function show(AuditLog $auditLog): View
    {
        // Check if user can view this audit log (must be from same organization)
        if ($auditLog->organization_id !== Auth::user()->organization_id) {
            abort(403, 'You can only view audit logs from your organization.');
        }

        return view('organization.audit-logs.show', compact('auditLog'));
    }

    /**
     * Export organization audit logs
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:csv,xlsx',
            'from_date' => 'nullable|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
        ]);

        $query = AuditLog::with(['user', 'organization'])
            ->forOrganization(Auth::user()->organization_id);

        // Apply date range filter
        if ($request->from_date && $request->to_date) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        $this->applyFilters($query, $request);

        // Limit export size for performance
        $maxRecords = config('audit.export.max_records', 50000);
        if ((clone $query)->count() > $maxRecords) {
            return back()->withErrors([
                'export' => "Export limited to {$maxRecords} records. Please narrow your date range."
            ]);
        }

        $orgName = Auth::user()->organization->name ?? 'organization';
        $filename = 'audit_logs_' . str_replace(' ', '_', $orgName) . '_' . now()->format('Y-m-d_H-i-s') . '.' . $request->format;

        return Excel::download(
            new AuditLogsExport($query),
            $filename
        );
    }

    /**
     * Get user activity comparison within organization
     */
    public function userActivityComparison(Request $request): JsonResponse
    {
        $days = $request->get('days', 30);
        $organizationId = Auth::user()->organization_id;
        
        $users = User::where('organization_id', $organizationId)
            ->with(['roles'])
            ->get();
            
        $comparison = [];
        
        foreach ($users as $user) {
            $activityCount = AuditLog::byUser($user->id)
                ->where('created_at', '>=', now()->subDays($days))
                ->count();
                
            $comparison[] = [
                'user_name' => $user->name,
                'user_email' => $user->email,
                'role' => $user->roles->first()?->name ?? 'No Role',
                'activity_count' => $activityCount,
                'last_activity' => AuditLog::byUser($user->id)
                    ->latest()
                    ->value('created_at')?->format('Y-m-d H:i:s') ?? 'Never',
            ];
        }

        return response()->json($comparison);
    }

    /**
     * Apply filters to the audit log query (organization-scoped)
     */
    private function applyFilters($query, Request $request): void
    {
        if ($request->filled('user_id')) {
            // Ensure user belongs to the same organization
            $user = User::where('id', $request->user_id)
                ->where('organization_id', Auth::user()->organization_id)
                ->first();
            if ($user) {
                $query->byUser($request->user_id);
            }
        }

        if ($request->filled('event_type')) {
            $query->byEventType($request->event_type);
        }

        if ($request->filled('category')) {
            $query->byCategory($request->category);
        }

        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        if ($request->filled('search')) {
            $query->search($request->search);
        }

        if ($request->filled('auditable_type')) {
            $query->where('auditable_type', 'like', '%' . $request->auditable_type . '%');
        }
    }
}