<?php

namespace App\Http\Controllers\Organization;

use App\Http\Controllers\Controller;
use App\Models\SubscriptionPlan;
use App\Services\SubscriptionService;
use App\Services\UsageTrackingService;
use App\Services\BillingService;
use Illuminate\Http\Request;

class SubscriptionController extends Controller
{
    protected $subscriptionService;
    protected $usageService;
    protected $billingService;

    public function __construct(
        SubscriptionService $subscriptionService,
        UsageTrackingService $usageService,
        BillingService $billingService
    ) {
        $this->middleware('auth');
        $this->subscriptionService = $subscriptionService;
        $this->usageService = $usageService;
        $this->billingService = $billingService;
    }

    /**
     * Show subscription details
     */
    public function show()
    {
        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;

        if (!$subscription) {
            return redirect()->route('subscription.plans');
        }

        $summary = $this->subscriptionService->getSubscriptionSummary($organization);
        $usage = $this->usageService->getUsageSummary($organization);
        $billing = $this->billingService->getBillingSummary($organization);

        return view('organization.subscription.show', compact('subscription', 'summary', 'usage', 'billing'));
    }

    /**
     * Show available plans
     */
    public function plans()
    {
        $organization = auth()->user()->organization;
        $currentSubscription = $organization->subscription;
        $plans = SubscriptionPlan::active()->ordered()->get();

        return view('organization.subscription.plans', compact('plans', 'currentSubscription'));
    }

    /**
     * Show upgrade form
     */
    public function upgrade(SubscriptionPlan $plan)
    {
        $organization = auth()->user()->organization;
        $currentSubscription = $organization->subscription;

        // Check if already on this plan
        if ($currentSubscription && $currentSubscription->subscription_plan_id == $plan->id) {
            return redirect()
                ->route('subscription.show')
                ->with('info', 'You are already on this plan');
        }

        return view('organization.subscription.upgrade', compact('plan', 'currentSubscription'));
    }

    /**
     * Process upgrade
     */
    public function processUpgrade(Request $request, SubscriptionPlan $plan)
    {
        $request->validate([
            'billing_cycle' => 'required|in:monthly,annual',
        ]);

        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;

        try {
            if ($subscription) {
                // Change existing plan
                $this->subscriptionService->changePlan($subscription, $plan, true);
                $message = 'Plan upgraded successfully';
            } else {
                // Create new subscription
                $subscription = $this->subscriptionService->createSubscription(
                    $organization,
                    $plan,
                    $request->billing_cycle,
                    false // No trial for upgrades
                );
                $message = 'Subscription created successfully';
            }

            // Generate invoice
            $invoice = $this->billingService->generateSubscriptionInvoice($subscription);

            return redirect()
                ->route('organization.payments.show', $invoice)
                ->with('success', $message . '. Please complete payment.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to process upgrade: ' . $e->getMessage());
        }
    }

    /**
     * Show cancel form
     */
    public function cancelForm()
    {
        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;

        if (!$subscription || $subscription->status === 'cancelled') {
            return redirect()->route('subscription.show');
        }

        return view('organization.subscription.cancel', compact('subscription'));
    }

    /**
     * Process cancellation
     */
    public function processCancel(Request $request)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
            'immediately' => 'boolean',
        ]);

        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;

        try {
            $this->subscriptionService->cancelSubscription(
                $subscription,
                $request->reason,
                $request->boolean('immediately')
            );

            $message = $request->boolean('immediately') ? 
                'Your subscription has been cancelled' : 
                'Your subscription will be cancelled at the end of the billing period';

            return redirect()
                ->route('subscription.show')
                ->with('success', $message);

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel subscription: ' . $e->getMessage());
        }
    }

    /**
     * Show trial expired page
     */
    public function trialExpired()
    {
        $organization = auth()->user()->organization;
        $plans = SubscriptionPlan::active()->ordered()->get();

        return view('organization.subscription.trial-expired', compact('plans'));
    }

    /**
     * Show subscription required page
     */
    public function required()
    {
        $plans = SubscriptionPlan::active()->ordered()->get();
        return view('organization.subscription.required', compact('plans'));
    }

    /**
     * Show subscription expired page
     */
    public function expired()
    {
        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;
        $plans = SubscriptionPlan::active()->ordered()->get();

        return view('organization.subscription.expired', compact('subscription', 'plans'));
    }

    /**
     * Show subscription suspended page
     */
    public function suspended()
    {
        $organization = auth()->user()->organization;
        $subscription = $organization->subscription;

        return view('organization.subscription.suspended', compact('subscription'));
    }
}
