<?php

namespace App\Http\Controllers\Organization;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Payment;
use App\Services\MpesaService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SubscriptionPaymentController extends Controller
{
    protected $mpesaService;

    public function __construct(MpesaService $mpesaService)
    {
        $this->mpesaService = $mpesaService;
    }

    /**
     * Display list of invoices and payments
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $organization = auth()->user()->organization;
        
        $invoices = Invoice::where('organization_id', $organization->id)
            ->with('payments')
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        $billingService = app(\App\Services\BillingService::class);
        $summary = $billingService->getBillingSummary($organization);

        return view('organization.payments.index', compact('invoices', 'summary'));
    }

    /**
     * Show payment page for invoice
     *
     * @param Invoice $invoice
     * @return \Illuminate\View\View
     */
    public function show(Invoice $invoice)
    {
        // Ensure user can only pay for their organization's invoices
        $this->authorize('view', $invoice);

        return view('organization.payments.show', compact('invoice'));
    }

    /**
     * Initiate M-Pesa payment
     *
     * @param Request $request
     * @param Invoice $invoice
     * @return \Illuminate\Http\RedirectResponse
     */
    public function initiate(Request $request, Invoice $invoice)
    {
        // Ensure user can only pay for their organization's invoices
        $this->authorize('pay', $invoice);

        $request->validate([
            'phone_number' => [
                'required',
                'string',
                'regex:/^(\+?254|0)?[17]\d{8}$/',
            ],
        ], [
            'phone_number.regex' => 'Please enter a valid Kenyan phone number (e.g., 0712345678 or 254712345678)',
        ]);

        // Check if invoice is already paid
        if ($invoice->isPaid()) {
            return back()->with('error', 'This invoice has already been paid.');
        }

        // Initiate payment
        $result = $this->mpesaService->initiateSubscriptionPayment(
            $invoice,
            $request->phone_number
        );

        if ($result['success']) {
            return back()->with([
                'success' => $result['message'],
                'payment_id' => $result['payment_id'],
                'checkout_request_id' => $result['checkout_request_id'],
            ]);
        }

        return back()->with('error', $result['message']);
    }

    /**
     * Check payment status
     *
     * @param Payment $payment
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkStatus(Payment $payment)
    {
        // Ensure user can only check their organization's payments
        if ($payment->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        return response()->json([
            'status' => $payment->status,
            'is_completed' => $payment->isCompleted(),
            'is_pending' => $payment->isPending(),
            'is_failed' => $payment->isFailed(),
            'mpesa_receipt_number' => $payment->mpesa_receipt_number,
            'completed_at' => $payment->completed_at?->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Query M-Pesa transaction status
     *
     * @param Payment $payment
     * @return \Illuminate\Http\JsonResponse
     */
    public function queryStatus(Payment $payment)
    {
        // Ensure user can only query their organization's payments
        if ($payment->organization_id !== auth()->user()->organization_id) {
            abort(403);
        }

        if (!$payment->mpesa_checkout_request_id) {
            return response()->json([
                'success' => false,
                'message' => 'No M-Pesa transaction found for this payment',
            ]);
        }

        $result = $this->mpesaService->queryTransactionStatus(
            $payment->mpesa_checkout_request_id
        );

        return response()->json($result);
    }

    /**
     * Show payment history
     *
     * @return \Illuminate\View\View
     */
    public function history()
    {
        $organization = auth()->user()->organization;
        
        $payments = Payment::where('organization_id', $organization->id)
            ->with('invoice')
            ->latest()
            ->paginate(20);

        return view('organization.payments.history', compact('payments'));
    }
}
