<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Services\OrganizationMailService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class OrganizationEmailSettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:edit_organization_settings');
    }

    /**
     * Show email settings page
     */
    public function index()
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            abort(403, 'No organization found');
        }

        return view('settings.email-setup', compact('organization'));
    }

    /**
     * Update email settings
     */
    public function update(Request $request)
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return redirect()->back()->with('error', 'No organization found');
        }

        $validator = Validator::make($request->all(), [
            'use_custom_mail' => 'required|boolean',
            'mail_driver' => 'required_if:use_custom_mail,1|in:smtp,sendmail,mailgun,ses,postmark',
            'mail_host' => 'required_if:use_custom_mail,1|string|max:255',
            'mail_port' => 'required_if:use_custom_mail,1|integer|min:1|max:65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'required_if:use_custom_mail,1|in:tls,ssl,none',
            'mail_from_address' => 'required_if:use_custom_mail,1|email|max:255',
            'mail_from_name' => 'required_if:use_custom_mail,1|string|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = [
            'use_custom_mail' => $request->use_custom_mail,
        ];

        if ($request->use_custom_mail) {
            $data = array_merge($data, [
                'mail_driver' => $request->mail_driver,
                'mail_host' => $request->mail_host,
                'mail_port' => $request->mail_port,
                'mail_username' => $request->mail_username,
                'mail_encryption' => $request->mail_encryption === 'none' ? null : $request->mail_encryption,
                'mail_from_address' => $request->mail_from_address,
                'mail_from_name' => $request->mail_from_name,
            ]);

            // Only update password if provided
            if ($request->filled('mail_password')) {
                $data['mail_password'] = $request->mail_password;
            }
        }

        $organization->update($data);

        return redirect()->back()->with('success', 'Email settings updated successfully!');
    }

    /**
     * Test email configuration
     */
    public function test(Request $request)
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return response()->json(['success' => false, 'message' => 'No organization found'], 403);
        }

        $validator = Validator::make($request->all(), [
            'test_email' => 'required|email',
            'use_custom_mail' => 'required|boolean',
            'mail_host' => 'required_if:use_custom_mail,1|string',
            'mail_port' => 'required_if:use_custom_mail,1|integer',
            'mail_encryption' => 'required_if:use_custom_mail,1|string',
            'mail_from_address' => 'required_if:use_custom_mail,1|email',
            'mail_from_name' => 'required_if:use_custom_mail,1|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Please fill in all required fields: ' . implode(', ', $validator->errors()->all())
            ], 422);
        }

        // Create a temporary organization object with test settings
        $testOrg = clone $organization;
        $testOrg->use_custom_mail = $request->use_custom_mail;
        
        if ($request->use_custom_mail) {
            $testOrg->mail_driver = $request->mail_driver ?? 'smtp';
            $testOrg->mail_host = $request->mail_host;
            $testOrg->mail_port = $request->mail_port;
            $testOrg->mail_username = $request->mail_username;
            $testOrg->mail_encryption = $request->mail_encryption === 'none' ? null : $request->mail_encryption;
            $testOrg->mail_from_address = $request->mail_from_address;
            $testOrg->mail_from_name = $request->mail_from_name;
            
            // Use provided password or existing one
            if ($request->filled('mail_password')) {
                $testOrg->mail_password = $request->mail_password;
            }
        }

        try {
            $success = OrganizationMailService::testConfiguration($testOrg, $request->test_email);

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'Test email sent successfully! Please check your inbox.'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send test email. Please check your configuration and try again.'
                ], 500);
            }
        } catch (\Exception $e) {
            \Log::error('Email test error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
}
