<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class OrganizationSettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'tenant_scope']);
        $this->middleware('permission:view_organization_settings')->only(['show']);
        $this->middleware('permission:edit_organization_settings')->only(['update']);
    }

    /**
     * Show organization settings
     */
    public function show()
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            abort(404, 'Organization not found');
        }

        return view('settings.organization', compact('organization'));
    }

    /**
     * Update organization settings
     */
    public function update(Request $request)
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            abort(404, 'Organization not found');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'settings.company_address' => 'nullable|string',
            'settings.company_phone' => 'nullable|string',
            'settings.company_email' => 'nullable|email',
            'settings.company_website' => 'nullable|url',
            'settings.working_hours_start' => 'nullable|string',
            'settings.working_hours_end' => 'nullable|string',
            'settings.timezone' => 'nullable|string',
        ]);

        $data = [
            'name' => $request->name,
        ];

        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($organization->logo) {
                Storage::disk('public')->delete($organization->logo);
            }
            
            $logoPath = $request->file('logo')->store('organizations/logos', 'public');
            $data['logo'] = $logoPath;
        }

        // Update settings
        $settings = $organization->settings ?? [];
        $settings['company_address'] = $request->input('settings.company_address');
        $settings['company_phone'] = $request->input('settings.company_phone');
        $settings['company_email'] = $request->input('settings.company_email');
        $settings['company_website'] = $request->input('settings.company_website');
        $settings['working_hours_start'] = $request->input('settings.working_hours_start');
        $settings['working_hours_end'] = $request->input('settings.working_hours_end');
        $settings['timezone'] = $request->input('settings.timezone');

        $data['settings'] = $settings;

        $organization->update($data);

        return redirect()->route('settings.organization')
            ->with('success', 'Organization settings updated successfully.');
    }
}