<?php

namespace App\Http\Controllers;

use App\Models\P9Form;
use App\Models\Employee;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class P9FormController extends Controller
{
    public function index(Request $request)
    {
        $query = P9Form::with('employee');

        if ($request->filled('year')) {
            $query->where('year', $request->year);
        }

        if ($request->filled('employee')) {
            $query->where('employee_id', $request->employee);
        }

        $p9Forms = $query->latest()->paginate(15);
        $employees = Employee::where('status', 'active')->get();
        $years = range(date('Y'), date('Y') - 5); // Current year and 5 years back

        return view('p9-forms.index', compact('p9Forms', 'employees', 'years'));
    }

    public function generate(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'year' => 'required|integer|min:2020|max:' . date('Y'),
        ]);

        try {
            $p9Form = P9Form::generateForEmployee($validated['employee_id'], $validated['year']);
            
            return redirect()->route('p9-forms.show', $p9Form)
                ->with('success', 'P9 Form generated successfully!');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    public function bulkGenerate(Request $request)
    {
        $validated = $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
            'year' => 'required|integer|min:2020|max:' . date('Y'),
        ]);

        $generated = 0;
        $errors = [];

        foreach ($validated['employee_ids'] as $employeeId) {
            try {
                P9Form::generateForEmployee($employeeId, $validated['year']);
                $generated++;
            } catch (\Exception $e) {
                $employee = Employee::find($employeeId);
                $errors[] = "Failed to generate P9 for {$employee->full_name}: " . $e->getMessage();
            }
        }

        $message = "Generated {$generated} P9 forms successfully.";
        if (!empty($errors)) {
            $message .= " Errors: " . implode(', ', $errors);
        }

        return redirect()->route('p9-forms.index')->with('success', $message);
    }

    public function show(P9Form $p9Form)
    {
        $p9Form->load('employee');
        return view('p9-forms.show', compact('p9Form'));
    }

    public function downloadPdf(P9Form $p9Form)
    {
        $p9Form->load('employee');
        
        $pdf = Pdf::loadView('p9-forms.pdf', compact('p9Form'));
        
        $filename = "P9_Form_{$p9Form->employee->full_name}_{$p9Form->year}.pdf";
        
        return $pdf->download($filename);
    }

    public function create()
    {
        $employees = Employee::where('status', 'active')->get();
        $years = range(date('Y'), date('Y') - 5);
        
        return view('p9-forms.create', compact('employees', 'years'));
    }

    public function bulkCreate()
    {
        $employees = Employee::where('status', 'active')->with('department')->get();
        $years = range(date('Y'), date('Y') - 5);
        
        return view('p9-forms.bulk-create', compact('employees', 'years'));
    }
}