<?php

namespace App\Http\Controllers;

use App\Models\Payroll;
use App\Models\Employee;
use App\Services\PayrollService;
use App\Services\KenyaTaxCalculator;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Carbon\Carbon;

class PayrollController extends Controller
{
    protected PayrollService $payrollService;

    public function __construct(PayrollService $payrollService)
    {
        $this->payrollService = $payrollService;
    }

    /**
     * Display a listing of payrolls
     */
    public function index(Request $request): JsonResponse
    {
        $query = Payroll::with(['employee', 'contract']);

        // Filter by year and month
        if ($request->has('year')) {
            $query->where('year', $request->year);
        }
        if ($request->has('month')) {
            $query->where('month', $request->month);
        }

        // Filter by employment type
        if ($request->has('employment_type')) {
            $query->where('employment_type', $request->employment_type);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by organization
        if ($request->has('organization_id')) {
            $query->where('organization_id', $request->organization_id);
        }

        $payrolls = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json($payrolls);
    }

    /**
     * Generate payroll for a specific month
     */
    public function generateMonthly(Request $request): JsonResponse
    {
        $request->validate([
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'employee_ids' => 'sometimes|array',
            'employee_ids.*' => 'exists:employees,id',
        ]);

        try {
            $payrolls = $this->payrollService->generateMonthlyPayroll(
                $request->year,
                $request->month,
                $request->employee_ids ?? []
            );

            return response()->json([
                'message' => 'Payroll generated successfully',
                'count' => $payrolls->count(),
                'payrolls' => $payrolls,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to generate payroll',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate payroll for a specific employee
     */
    public function generateEmployee(Request $request, Employee $employee): JsonResponse
    {
        $request->validate([
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
        ]);

        try {
            $payroll = $this->payrollService->generateEmployeePayroll(
                $employee,
                $request->year,
                $request->month
            );

            return response()->json([
                'message' => 'Employee payroll generated successfully',
                'payroll' => $payroll->load(['employee', 'contract']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to generate employee payroll',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Show a specific payroll
     */
    public function show(Payroll $payroll): JsonResponse
    {
        return response()->json($payroll->load(['employee', 'contract', 'attendances']));
    }

    /**
     * Calculate payroll preview without saving
     */
    public function preview(Request $request): JsonResponse
    {
        $request->validate([
            'employment_type' => 'required|string|in:permanent,contract,freelance,casual,intern,part_time',
            'basic_salary' => 'required|numeric|min:0',
            'house_allowance' => 'sometimes|numeric|min:0',
            'transport_allowance' => 'sometimes|numeric|min:0',
            'medical_allowance' => 'sometimes|numeric|min:0',
            'other_allowances' => 'sometimes|numeric|min:0',
            'bonuses' => 'sometimes|numeric|min:0',
            'overtime_pay' => 'sometimes|numeric|min:0',
            'helb_deduction' => 'sometimes|numeric|min:0',
            'loan_deductions' => 'sometimes|numeric|min:0',
            'advance_deductions' => 'sometimes|numeric|min:0',
            'other_deductions' => 'sometimes|numeric|min:0',
            'statutory_deductions_applicable' => 'sometimes|boolean',
            'withholding_tax_rate' => 'sometimes|numeric|min:0|max:1',
        ]);

        $salaryComponents = $request->only([
            'basic_salary', 'house_allowance', 'transport_allowance', 
            'medical_allowance', 'other_allowances', 'bonuses', 'overtime_pay',
            'helb_deduction', 'loan_deductions', 'advance_deductions', 'other_deductions',
            'statutory_deductions_applicable', 'withholding_tax_rate'
        ]);

        $calculations = KenyaTaxCalculator::calculatePayrollByEmploymentType(
            $salaryComponents,
            $request->employment_type
        );

        // Add employment type rules for reference
        $taxRules = KenyaTaxCalculator::getEmploymentTypeTaxRules($request->employment_type);

        return response()->json([
            'employment_type' => $request->employment_type,
            'calculations' => $calculations,
            'tax_rules' => $taxRules,
        ]);
    }

    /**
     * Approve a payroll
     */
    public function approve(Request $request, Payroll $payroll): JsonResponse
    {
        if ($payroll->status !== 'draft') {
            return response()->json([
                'error' => 'Only draft payrolls can be approved'
            ], 400);
        }

        try {
            $this->payrollService->approvePayroll($payroll, $request->user()->id);

            return response()->json([
                'message' => 'Payroll approved successfully',
                'payroll' => $payroll->fresh(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to approve payroll',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Process an approved payroll
     */
    public function process(Request $request, Payroll $payroll): JsonResponse
    {
        try {
            $this->payrollService->processPayroll($payroll, $request->user()->id);

            return response()->json([
                'message' => 'Payroll processed successfully',
                'payroll' => $payroll->fresh(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to process payroll',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get payroll summary for a period
     */
    public function summary(Request $request): JsonResponse
    {
        $request->validate([
            'year' => 'required|integer|min:2020|max:2030',
            'month' => 'required|integer|min:1|max:12',
            'organization_id' => 'sometimes|exists:organizations,id',
        ]);

        $summary = $this->payrollService->generatePayrollSummary(
            $request->year,
            $request->month,
            $request->organization_id
        );

        return response()->json($summary);
    }

    /**
     * Get employment type tax rules
     */
    public function taxRules(Request $request): JsonResponse
    {
        $request->validate([
            'employment_type' => 'required|string|in:permanent,contract,freelance,casual,intern,part_time',
        ]);

        $rules = KenyaTaxCalculator::getEmploymentTypeTaxRules($request->employment_type);

        return response()->json([
            'employment_type' => $request->employment_type,
            'rules' => $rules,
        ]);
    }

    /**
     * Calculate hourly payroll
     */
    public function calculateHourly(Request $request): JsonResponse
    {
        $request->validate([
            'hourly_rate' => 'required|numeric|min:0',
            'regular_hours' => 'required|numeric|min:0',
            'overtime_hours' => 'sometimes|numeric|min:0',
            'overtime_multiplier' => 'sometimes|numeric|min:1',
            'house_allowance' => 'sometimes|numeric|min:0',
            'transport_allowance' => 'sometimes|numeric|min:0',
            'other_allowances' => 'sometimes|numeric|min:0',
        ]);

        $additionalComponents = $request->only([
            'house_allowance', 'transport_allowance', 'other_allowances'
        ]);

        $calculations = KenyaTaxCalculator::calculateHourlyPayroll(
            $request->hourly_rate,
            $request->regular_hours,
            $request->overtime_hours ?? 0,
            $request->overtime_multiplier ?? 1.5,
            $additionalComponents
        );

        return response()->json($calculations);
    }

    /**
     * Calculate daily payroll
     */
    public function calculateDaily(Request $request): JsonResponse
    {
        $request->validate([
            'daily_rate' => 'required|numeric|min:0',
            'days_worked' => 'required|integer|min:0',
            'house_allowance' => 'sometimes|numeric|min:0',
            'transport_allowance' => 'sometimes|numeric|min:0',
            'other_allowances' => 'sometimes|numeric|min:0',
        ]);

        $additionalComponents = $request->only([
            'house_allowance', 'transport_allowance', 'other_allowances'
        ]);

        $calculations = KenyaTaxCalculator::calculateDailyPayroll(
            $request->daily_rate,
            $request->days_worked,
            $additionalComponents
        );

        return response()->json($calculations);
    }

    /**
     * Update payroll
     */
    public function update(Request $request, Payroll $payroll): JsonResponse
    {
        if ($payroll->status !== 'draft') {
            return response()->json([
                'error' => 'Only draft payrolls can be updated'
            ], 400);
        }

        $request->validate([
            'bonuses' => 'sometimes|numeric|min:0',
            'performance_bonus' => 'sometimes|numeric|min:0',
            'merit_increase' => 'sometimes|numeric|min:0',
            'commission' => 'sometimes|numeric|min:0',
            'other_allowances' => 'sometimes|numeric|min:0',
            'loan_deductions' => 'sometimes|numeric|min:0',
            'advance_deductions' => 'sometimes|numeric|min:0',
            'disciplinary_deductions' => 'sometimes|numeric|min:0',
            'other_deductions' => 'sometimes|numeric|min:0',
        ]);

        $payroll->update($request->only([
            'bonuses', 'performance_bonus', 'merit_increase', 'commission',
            'other_allowances', 'loan_deductions', 'advance_deductions',
            'disciplinary_deductions', 'other_deductions'
        ]));

        // Recalculate payroll with updated values
        $contract = $payroll->contract;
        $salaryComponents = [
            'basic_salary' => $payroll->contract_basic_pay,
            'house_allowance' => $payroll->house_allowance,
            'transport_allowance' => $payroll->transport_allowance,
            'medical_allowance' => $payroll->medical_allowance,
            'other_allowances' => $payroll->other_allowances,
            'bonuses' => $payroll->bonuses,
            'performance_bonus' => $payroll->performance_bonus,
            'merit_increase' => $payroll->merit_increase,
            'commission' => $payroll->commission,
            'overtime_pay' => $payroll->overtime_pay,
            'helb_deduction' => $payroll->helb_deduction,
            'loan_deductions' => $payroll->loan_deductions,
            'advance_deductions' => $payroll->advance_deductions,
            'disciplinary_deductions' => $payroll->disciplinary_deductions,
            'other_deductions' => $payroll->other_deductions,
            'statutory_deductions_applicable' => $contract->statutory_deductions_applicable,
            'withholding_tax_rate' => $contract->withholding_tax_rate,
        ];

        $calculations = KenyaTaxCalculator::calculatePayrollByEmploymentType(
            $salaryComponents,
            $payroll->employment_type
        );

        // Update calculated fields
        foreach ($calculations as $field => $value) {
            if (in_array($field, $payroll->getFillable())) {
                $payroll->$field = $value;
            }
        }

        $payroll->save();

        return response()->json([
            'message' => 'Payroll updated successfully',
            'payroll' => $payroll->fresh()->load(['employee', 'contract']),
        ]);
    }

    /**
     * Delete a payroll (only drafts)
     */
    public function destroy(Payroll $payroll): JsonResponse
    {
        if ($payroll->status !== 'draft') {
            return response()->json([
                'error' => 'Only draft payrolls can be deleted'
            ], 400);
        }

        $payroll->delete();

        return response()->json([
            'message' => 'Payroll deleted successfully'
        ]);
    }
}