<?php

namespace App\Http\Controllers;

use App\Models\PayrollSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PayrollSettingsController extends Controller
{
    /**
     * Display the payroll settings.
     */
    public function index()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific payroll settings.');
        }
        
        $settings = PayrollSettings::getForOrganization($user->organization_id);
        
        $this->authorize('view', $settings);
        
        return view('payroll.settings.index', compact('settings'));
    }

    /**
     * Show the form for editing payroll settings.
     */
    public function edit()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific payroll settings.');
        }
        
        $settings = PayrollSettings::getForOrganization($user->organization_id);
        
        $this->authorize('update', $settings);
        
        return view('payroll.settings.edit', compact('settings'));
    }

    /**
     * Update the payroll settings.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific payroll settings.');
        }
        
        $settings = PayrollSettings::getForOrganization($user->organization_id);

        $this->authorize('update', $settings);

        $validated = $request->validate([
            'personal_relief' => 'required|numeric|min:0',
            'insurance_relief_rate' => 'required|numeric|min:0|max:100',
            'insurance_relief_max' => 'required|numeric|min:0',
            'pension_relief_max' => 'required|numeric|min:0',
            'pension_relief_rate' => 'required|numeric|min:0|max:100',
            'shif_rate' => 'required|numeric|min:0|max:100',
            'shif_min_contribution' => 'required|numeric|min:0',
            'nssf_employee_rate' => 'required|numeric|min:0|max:100',
            'nssf_employer_rate' => 'required|numeric|min:0|max:100',
            'nssf_lower_limit' => 'required|numeric|min:0',
            'nssf_upper_limit' => 'required|numeric|min:0',
            'housing_levy_employee_rate' => 'required|numeric|min:0|max:100',
            'housing_levy_employer_rate' => 'required|numeric|min:0|max:100',
            'default_house_allowance_rate' => 'required|numeric|min:0|max:100',
            'default_transport_allowance_rate' => 'required|numeric|min:0|max:100',
            'max_transport_allowance' => 'required|numeric|min:0',
            'default_medical_allowance' => 'required|numeric|min:0',
            'default_monthly_working_days' => 'required|integer|min:1|max:31',
            'default_weekly_working_days' => 'required|integer|min:1|max:7',
            'default_hours_per_day' => 'required|integer|min:1|max:24',
            'kra_submission_deadline' => 'required|integer|min:1|max:31',
            'shif_submission_deadline' => 'required|integer|min:1|max:31',
            'nssf_submission_deadline' => 'required|integer|min:1|max:31',
            'housing_levy_submission_deadline' => 'required|integer|min:1|max:31',
        ]);

        // Convert percentages to decimals
        $validated['insurance_relief_rate'] = $validated['insurance_relief_rate'] / 100;
        $validated['pension_relief_rate'] = $validated['pension_relief_rate'] / 100;
        $validated['shif_rate'] = $validated['shif_rate'] / 100;
        $validated['nssf_employee_rate'] = $validated['nssf_employee_rate'] / 100;
        $validated['nssf_employer_rate'] = $validated['nssf_employer_rate'] / 100;
        $validated['housing_levy_employee_rate'] = $validated['housing_levy_employee_rate'] / 100;
        $validated['housing_levy_employer_rate'] = $validated['housing_levy_employer_rate'] / 100;
        $validated['default_house_allowance_rate'] = $validated['default_house_allowance_rate'] / 100;
        $validated['default_transport_allowance_rate'] = $validated['default_transport_allowance_rate'] / 100;

        $settings->update($validated);

        return redirect()->route('payroll.settings.index')
            ->with('success', 'Payroll settings updated successfully.');
    }

    /**
     * Reset payroll settings to defaults.
     */
    public function reset()
    {
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific payroll settings.');
        }
        
        $settings = PayrollSettings::getForOrganization($user->organization_id);

        $this->authorize('reset', $settings);

        $settings->update([
            'paye_bands' => PayrollSettings::getDefaultPayeBands(),
            'personal_relief' => 2400,
            'insurance_relief_rate' => 0.15,
            'insurance_relief_max' => 5000,
            'pension_relief_max' => 20000,
            'pension_relief_rate' => 0.30,
            'shif_rate' => 0.0275,
            'shif_min_contribution' => 300,
            'nssf_employee_rate' => 0.06,
            'nssf_employer_rate' => 0.06,
            'nssf_lower_limit' => 9000,
            'nssf_upper_limit' => 108000,
            'housing_levy_employee_rate' => 0.015,
            'housing_levy_employer_rate' => 0.015,
            'default_house_allowance_rate' => 0.15,
            'default_transport_allowance_rate' => 0.05,
            'max_transport_allowance' => 5000,
            'default_medical_allowance' => 2000,
            'default_monthly_working_days' => 22,
            'default_weekly_working_days' => 5,
            'default_hours_per_day' => 8,
            'kra_submission_deadline' => 9,
            'shif_submission_deadline' => 15,
            'nssf_submission_deadline' => 15,
            'housing_levy_submission_deadline' => 9,
            'config_version' => '2026.02',
            'last_regulatory_update' => now(),
        ]);

        return redirect()->route('payroll.settings.index')
            ->with('success', 'Payroll settings reset to defaults.');
    }

    /**
     * Test payroll calculations with sample data.
     */
    public function testCalculations(Request $request)
    {
        $this->authorize('testCalculations', PayrollSettings::class);
        
        $user = Auth::user();
        
        // System admins don't have organization-specific settings
        if (!$user->organization_id) {
            abort(403, 'System administrators cannot access organization-specific payroll settings.');
        }
        
        $settings = PayrollSettings::getForOrganization($user->organization_id);
        
        $grossSalary = $request->input('gross_salary', 50000);
        
        $calculations = [
            'gross_salary' => $grossSalary,
            'paye_tax' => $settings->calculatePayeTax($grossSalary - $settings->personal_relief),
            'shif_contribution' => $settings->calculateShifContribution($grossSalary),
            'nssf_employee' => $settings->calculateNssfContribution($grossSalary, true),
            'nssf_employer' => $settings->calculateNssfContribution($grossSalary, false),
            'housing_levy_employee' => $settings->calculateHousingLevyContribution($grossSalary, true),
            'housing_levy_employer' => $settings->calculateHousingLevyContribution($grossSalary, false),
        ];
        
        $calculations['total_deductions'] = $calculations['paye_tax'] + 
                                          $calculations['shif_contribution'] + 
                                          $calculations['nssf_employee'] + 
                                          $calculations['housing_levy_employee'];
        
        $calculations['net_salary'] = $calculations['gross_salary'] - $calculations['total_deductions'];
        
        return response()->json($calculations);
    }
}