<?php

namespace App\Http\Controllers;

use App\Models\PerformanceReview;
use App\Models\Employee;
use Illuminate\Http\Request;

class PerformanceReviewController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', PerformanceReview::class);
        
        $query = PerformanceReview::with(['employee', 'reviewer']);

        // Search by employee
        if ($request->filled('employee')) {
            $query->where('employee_id', $request->employee);
        }

        // Filter by period
        if ($request->filled('period')) {
            $query->where('review_period', $request->period);
        }

        // Filter by review type
        if ($request->filled('review_type')) {
            $query->where('review_type', $request->review_type);
        }

        // Filter by rating
        if ($request->filled('rating')) {
            $query->where('overall_rating', '>=', $request->rating)
                  ->where('overall_rating', '<', $request->rating + 1);
        }

        // Filter by year
        if ($request->filled('year')) {
            $query->whereYear('review_date', $request->year);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $reviews = $query->orderBy('review_date', 'desc')->paginate(15);

        // Enhanced Statistics
        $stats = [
            'total_reviews' => PerformanceReview::count(),
            'average_rating' => round(PerformanceReview::avg('overall_rating'), 1),
            'excellent_count' => PerformanceReview::where('overall_rating', '>=', 4.5)->count(),
            'needs_improvement' => PerformanceReview::where('overall_rating', '<', 3)->count(),
            'pending_reviews' => PerformanceReview::where('status', 'draft')->count(),
            'overdue_reviews' => PerformanceReview::whereNotNull('due_date')
                ->where('due_date', '<', now())
                ->where('status', '!=', 'completed')
                ->count(),
        ];

        $employees = Employee::where('status', 'active')->get();
        $years = range(date('Y'), date('Y') - 5);

        return view('performance.index', compact('reviews', 'stats', 'employees', 'years'));
    }

    public function create()
    {
        $this->authorize('create', PerformanceReview::class);
        
        $employees = Employee::where('status', 'active')->get();
        $reviewers = Employee::where('status', 'active')->get();

        return view('performance.create', compact('employees', 'reviewers'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', PerformanceReview::class);
        
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'reviewer_id' => 'required|exists:users,id',
            'review_date' => 'required|date',
            'due_date' => 'nullable|date|after:review_date',
            'review_period' => 'required|in:Q1,Q2,Q3,Q4,Yearly,Mid-Year',
            'review_type' => 'required|in:annual,mid_year,quarterly,probation,project_based',
            'is_self_assessment' => 'boolean',
            'performance_ratings' => 'required|array',
            'performance_ratings.quality_of_work' => 'required|numeric|min:1|max:5',
            'performance_ratings.productivity' => 'required|numeric|min:1|max:5',
            'performance_ratings.communication' => 'required|numeric|min:1|max:5',
            'performance_ratings.teamwork' => 'required|numeric|min:1|max:5',
            'performance_ratings.initiative' => 'required|numeric|min:1|max:5',
            'performance_ratings.attendance_punctuality' => 'required|numeric|min:1|max:5',
            'overall_rating' => 'required|numeric|min:1|max:5',
            'strengths' => 'nullable|string',
            'areas_of_improvement' => 'nullable|string',
            'goals' => 'nullable|string',
            'comments' => 'nullable|string',
            'manager_comments' => 'nullable|string',
            'development_areas' => 'nullable|array',
        ]);

        $validated['organization_id'] = auth()->user()->organization_id;
        $validated['status'] = 'draft';

        PerformanceReview::create($validated);

        return redirect()->route('performance.index')
            ->with('success', 'Performance review created successfully.');
    }

    public function show(PerformanceReview $performance)
    {
        $this->authorize('view', $performance);
        
        $performance->load([
            'employee.department', 
            'employee.position', 
            'reviewer',
            'goals.progressUpdates',
            'selfAssessment',
            'competencyRatings.performanceCompetency'
        ]);

        // Get previous reviews for comparison
        $previousReviews = PerformanceReview::where('employee_id', $performance->employee_id)
            ->where('id', '!=', $performance->id)
            ->orderBy('review_date', 'desc')
            ->limit(3)
            ->get();

        // Calculate goal achievement if goals exist
        if ($performance->goals && $performance->goals->isNotEmpty()) {
            $goalAchievementScore = $performance->calculateGoalAchievementScore();
            $performance->update(['goal_achievement_score' => $goalAchievementScore]);
        }

        return view('performance.show', compact('performance', 'previousReviews'));
    }

    public function edit(PerformanceReview $performance)
    {
        $this->authorize('update', $performance);
        
        $employees = Employee::where('status', 'active')->get();
        $reviewers = Employee::where('status', 'active')->get();

        return view('performance.edit', compact('performance', 'employees', 'reviewers'));
    }

    public function update(Request $request, PerformanceReview $performance)
    {
        $this->authorize('update', $performance);
        
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'reviewer_id' => 'required|exists:users,id',
            'review_date' => 'required|date',
            'due_date' => 'nullable|date|after:review_date',
            'review_period' => 'required|in:Q1,Q2,Q3,Q4,Yearly,Mid-Year',
            'review_type' => 'required|in:annual,mid_year,quarterly,probation,project_based',
            'is_self_assessment' => 'boolean',
            'performance_ratings' => 'required|array',
            'performance_ratings.quality_of_work' => 'required|numeric|min:1|max:5',
            'performance_ratings.productivity' => 'required|numeric|min:1|max:5',
            'performance_ratings.communication' => 'required|numeric|min:1|max:5',
            'performance_ratings.teamwork' => 'required|numeric|min:1|max:5',
            'performance_ratings.initiative' => 'required|numeric|min:1|max:5',
            'performance_ratings.attendance_punctuality' => 'required|numeric|min:1|max:5',
            'overall_rating' => 'required|numeric|min:1|max:5',
            'strengths' => 'nullable|string',
            'areas_of_improvement' => 'nullable|string',
            'goals' => 'nullable|string',
            'comments' => 'nullable|string',
            'manager_comments' => 'nullable|string',
            'development_areas' => 'nullable|array',
            'status' => 'required|in:draft,submitted,acknowledged,completed',
        ]);

        $performance->update($validated);

        return redirect()->route('performance.show', $performance)
            ->with('success', 'Performance review updated successfully.');
    }

    public function destroy(PerformanceReview $performance)
    {
        $this->authorize('delete', $performance);
        
        $performance->delete();

        return redirect()->route('performance.index')
            ->with('success', 'Performance review deleted successfully.');
    }
}
