<?php

namespace App\Http\Controllers;

use App\Models\PerformanceSelfAssessment;
use App\Models\Employee;
use App\Models\PerformanceReview;
use Illuminate\Http\Request;

class PerformanceSelfAssessmentController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', PerformanceSelfAssessment::class);
        
        $query = PerformanceSelfAssessment::with(['employee']);

        // Filter by employee (for managers/HR)
        if ($request->filled('employee') && auth()->user()->can('view_all_assessments')) {
            $query->where('employee_id', $request->employee);
        } elseif (auth()->user()->employee) {
            // Employees can only see their own assessments
            $query->where('employee_id', auth()->user()->employee->id);
        }

        if ($request->filled('period')) {
            $query->where('period', $request->period);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $assessments = $query->orderBy('assessment_date', 'desc')->paginate(15);

        // Statistics
        $stats = [
            'total_assessments' => PerformanceSelfAssessment::count(),
            'submitted_assessments' => PerformanceSelfAssessment::where('status', 'submitted')->count(),
            'draft_assessments' => PerformanceSelfAssessment::where('status', 'draft')->count(),
            'average_satisfaction' => round(PerformanceSelfAssessment::avg('overall_satisfaction'), 1),
        ];

        $employees = Employee::where('status', 'active')->get();

        return view('performance.self-assessments.index', compact('assessments', 'stats', 'employees'));
    }

    public function create(Request $request)
    {
        $this->authorize('create', PerformanceSelfAssessment::class);
        
        // Employees can only create their own assessments
        $employee = auth()->user()->employee;
        if (!$employee && !auth()->user()->can('create_assessments_for_others')) {
            abort(403, 'Only employees can create self-assessments.');
        }

        $employees = auth()->user()->can('create_assessments_for_others') 
            ? Employee::where('status', 'active')->get() 
            : collect([$employee]);

        $performanceReviews = PerformanceReview::with('employee')->get();

        return view('performance.self-assessments.create', compact('employees', 'performanceReviews'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', PerformanceSelfAssessment::class);
        
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'performance_review_id' => 'nullable|exists:performance_reviews,id',
            'assessment_date' => 'required|date',
            'period' => 'required|in:Q1,Q2,Q3,Q4,Mid-Year,Annual',
            'achievements' => 'required|array|min:1',
            'achievements.*' => 'required|string',
            'challenges_faced' => 'required|array|min:1',
            'challenges_faced.*' => 'required|string',
            'strengths_identified' => 'required|string',
            'areas_for_development' => 'required|string',
            'learning_goals' => 'required|array|min:1',
            'learning_goals.*' => 'required|string',
            'career_aspirations' => 'required|array|min:1',
            'career_aspirations.*' => 'required|string',
            'support_needed' => 'required|string',
            'overall_satisfaction' => 'required|numeric|min:1|max:5',
            'additional_feedback' => 'nullable|string',
            'submit_now' => 'boolean',
        ]);

        // Ensure employee can only create their own assessment
        if (!auth()->user()->can('create_assessments_for_others') && 
            $validated['employee_id'] != auth()->user()->employee->id) {
            abort(403, 'You can only create your own self-assessment.');
        }

        $validated['organization_id'] = auth()->user()->organization_id;
        $validated['status'] = $request->boolean('submit_now') ? 'submitted' : 'draft';
        
        if ($request->boolean('submit_now')) {
            $validated['submitted_at'] = now();
        }

        $assessment = PerformanceSelfAssessment::create($validated);

        $message = $request->boolean('submit_now') 
            ? 'Self-assessment submitted successfully.' 
            : 'Self-assessment saved as draft.';

        return redirect()->route('performance.self-assessments.show', $assessment)
            ->with('success', $message);
    }

    public function show(PerformanceSelfAssessment $selfAssessment)
    {
        $this->authorize('view', $selfAssessment);
        
        $selfAssessment->load(['employee', 'performanceReview']);

        return view('performance.self-assessments.show', compact('selfAssessment'));
    }

    public function edit(PerformanceSelfAssessment $selfAssessment)
    {
        $this->authorize('update', $selfAssessment);
        
        // Only draft assessments can be edited
        if ($selfAssessment->status !== 'draft') {
            return redirect()->route('performance.self-assessments.show', $selfAssessment)
                ->with('error', 'Only draft assessments can be edited.');
        }

        $employees = auth()->user()->can('create_assessments_for_others') 
            ? Employee::where('status', 'active')->get() 
            : collect([auth()->user()->employee]);

        $performanceReviews = PerformanceReview::with('employee')->get();

        return view('performance.self-assessments.edit', compact('selfAssessment', 'employees', 'performanceReviews'));
    }

    public function update(Request $request, PerformanceSelfAssessment $selfAssessment)
    {
        $this->authorize('update', $selfAssessment);
        
        if ($selfAssessment->status !== 'draft') {
            return redirect()->route('performance.self-assessments.show', $selfAssessment)
                ->with('error', 'Only draft assessments can be updated.');
        }

        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'performance_review_id' => 'nullable|exists:performance_reviews,id',
            'assessment_date' => 'required|date',
            'period' => 'required|in:Q1,Q2,Q3,Q4,Mid-Year,Annual',
            'achievements' => 'required|array|min:1',
            'achievements.*' => 'required|string',
            'challenges_faced' => 'required|array|min:1',
            'challenges_faced.*' => 'required|string',
            'strengths_identified' => 'required|string',
            'areas_for_development' => 'required|string',
            'learning_goals' => 'required|array|min:1',
            'learning_goals.*' => 'required|string',
            'career_aspirations' => 'required|array|min:1',
            'career_aspirations.*' => 'required|string',
            'support_needed' => 'required|string',
            'overall_satisfaction' => 'required|numeric|min:1|max:5',
            'additional_feedback' => 'nullable|string',
            'submit_now' => 'boolean',
        ]);

        if ($request->boolean('submit_now')) {
            $validated['status'] = 'submitted';
            $validated['submitted_at'] = now();
        }

        $selfAssessment->update($validated);

        $message = $request->boolean('submit_now') 
            ? 'Self-assessment submitted successfully.' 
            : 'Self-assessment updated successfully.';

        return redirect()->route('performance.self-assessments.show', $selfAssessment)
            ->with('success', $message);
    }

    public function submit(PerformanceSelfAssessment $selfAssessment)
    {
        $this->authorize('update', $selfAssessment);
        
        if ($selfAssessment->status !== 'draft') {
            return redirect()->route('performance.self-assessments.show', $selfAssessment)
                ->with('error', 'Assessment has already been submitted.');
        }

        $selfAssessment->submit();

        return redirect()->route('performance.self-assessments.show', $selfAssessment)
            ->with('success', 'Self-assessment submitted successfully.');
    }

    public function destroy(PerformanceSelfAssessment $selfAssessment)
    {
        $this->authorize('delete', $selfAssessment);
        
        $selfAssessment->delete();

        return redirect()->route('performance.self-assessments.index')
            ->with('success', 'Self-assessment deleted successfully.');
    }
}