<?php

namespace App\Http\Controllers;

use Spatie\Permission\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PermissionController extends Controller
{
    public function index()
    {
        $this->authorize('viewAny', Permission::class);
        
        $query = Permission::query();
        
        // Org admins cannot see system admin permissions
        if (!Auth::user()->isSystemAdmin()) {
            $query->whereNotIn('name', [
                'manage_system_admin',
                'view_system_admin',
                'create_system_admin',
                'edit_system_admin',
                'delete_system_admin'
            ]);
        }
        
        $permissions = $query->get()->groupBy(function($permission) {
            $parts = explode('_', $permission->name);
            return count($parts) > 1 ? $parts[1] : 'general';
        });
        
        return view('permissions.index', compact('permissions'));
    }

    public function create()
    {
        $this->authorize('create', Permission::class);
        
        return view('permissions.create');
    }

    public function store(Request $request)
    {
        $this->authorize('create', Permission::class);
        
        $validated = $request->validate([
            'name' => 'required|string|unique:permissions|max:255',
        ]);

        Permission::create([
            'name' => $validated['name'],
            'guard_name' => 'web',
        ]);

        return redirect()->route('permissions.index')->with('success', 'Permission created successfully!');
    }

    public function edit(Permission $permission)
    {
        $this->authorize('update', $permission);
        
        return view('permissions.edit', compact('permission'));
    }

    public function update(Request $request, Permission $permission)
    {
        $this->authorize('update', $permission);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name,' . $permission->id,
        ]);

        $permission->update($validated);

        return redirect()->route('permissions.index')->with('success', 'Permission updated successfully!');
    }

    public function destroy(Permission $permission)
    {
        $this->authorize('delete', $permission);
        
        // Check if permission is assigned to any roles
        if ($permission->roles()->count() > 0) {
            return redirect()->route('permissions.index')->with('error', 'Cannot delete permission assigned to roles.');
        }

        $permission->delete();

        return redirect()->route('permissions.index')->with('success', 'Permission deleted successfully!');
    }
}
