<?php

namespace App\Http\Controllers;

use App\Models\Application;
use App\Models\JobPosting;
use App\Models\Interview;
use App\Models\JobOffer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RecruitmentAnalyticsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_applications');
    }

    public function index(Request $request)
    {
        $orgId = auth()->user()->organization_id;

        // Overall stats
        $stats = [
            'total_applications' => Application::where('organization_id', $orgId)->count(),
            'active_postings' => JobPosting::where('organization_id', $orgId)->where('status', 'active')->count(),
            'interviews_scheduled' => Interview::where('organization_id', $orgId)->where('status', 'scheduled')->count(),
            'offers_sent' => JobOffer::where('organization_id', $orgId)->where('status', 'sent')->count(),
            'offers_accepted' => JobOffer::where('organization_id', $orgId)->where('status', 'accepted')->count(),
        ];

        // Application funnel
        $funnel = Application::where('organization_id', $orgId)
            ->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        // Applications over time (last 30 days)
        $applicationsOverTime = Application::where('organization_id', $orgId)
            ->where('applied_at', '>=', now()->subDays(30))
            ->select(DB::raw('DATE(applied_at) as date'), DB::raw('count(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top performing job postings
        $topPostings = JobPosting::where('organization_id', $orgId)
            ->withCount('applications')
            ->orderBy('applications_count', 'desc')
            ->limit(5)
            ->get();

        // Time to hire (average days from application to hire)
        $avgTimeToHire = Application::where('organization_id', $orgId)
            ->whereNotNull('converted_to_employee_at')
            ->selectRaw('AVG(DATEDIFF(converted_to_employee_at, applied_at)) as avg_days')
            ->value('avg_days');

        // Conversion rates
        $conversionRates = [
            'application_to_interview' => $this->calculateConversionRate($orgId, 'interviewed'),
            'interview_to_offer' => $this->calculateConversionRate($orgId, 'offered'),
            'offer_to_hire' => $this->calculateOfferAcceptanceRate($orgId),
        ];

        return view('recruitment.analytics.dashboard', compact(
            'stats',
            'funnel',
            'applicationsOverTime',
            'topPostings',
            'avgTimeToHire',
            'conversionRates'
        ));
    }

    private function calculateConversionRate($orgId, $status)
    {
        $total = Application::where('organization_id', $orgId)->count();
        if ($total == 0) return 0;

        $converted = Application::where('organization_id', $orgId)
            ->whereIn('status', [$status, 'offered'])
            ->orWhereNotNull('converted_to_employee_at')
            ->count();

        return round(($converted / $total) * 100, 1);
    }

    private function calculateOfferAcceptanceRate($orgId)
    {
        $totalOffers = JobOffer::where('organization_id', $orgId)
            ->whereIn('status', ['sent', 'accepted', 'declined'])
            ->count();

        if ($totalOffers == 0) return 0;

        $acceptedOffers = JobOffer::where('organization_id', $orgId)
            ->where('status', 'accepted')
            ->count();

        return round(($acceptedOffers / $totalOffers) * 100, 1);
    }

    public function export(Request $request)
    {
        $orgId = auth()->user()->organization_id;

        $applications = Application::where('organization_id', $orgId)
            ->with(['jobPosting:id,title', 'jobPosting.department:id,name'])
            ->get();

        $filename = 'recruitment_report_' . now()->format('Y-m-d') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($applications) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['Name', 'Email', 'Position', 'Department', 'Status', 'Applied Date', 'Days Since Applied']);

            foreach ($applications as $app) {
                fputcsv($file, [
                    $app->full_name,
                    $app->email,
                    $app->jobPosting->title,
                    $app->jobPosting->department->name,
                    ucfirst($app->status),
                    $app->applied_at->format('Y-m-d'),
                    $app->applied_at->diffInDays(now()),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
