<?php

namespace App\Http\Controllers;

use App\Models\JobPosting;
use App\Models\Application;
use App\Models\Interview;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RecruitmentReportsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view_recruitment_reports');
    }

    /**
     * Display recruitment analytics dashboard
     */
    public function index(Request $request)
    {
        $dateRange = $request->get('date_range', '30'); // Default 30 days
        $startDate = Carbon::now()->subDays($dateRange);
        $endDate = Carbon::now();

        // Key Metrics
        $metrics = [
            'total_job_postings' => JobPosting::count(),
            'active_job_postings' => JobPosting::active()->count(),
            'total_applications' => Application::count(),
            'applications_this_period' => Application::where('applied_at', '>=', $startDate)->count(),
            'interviews_scheduled' => Interview::where('status', 'scheduled')->count(),
            'offers_extended' => Application::where('status', 'offered')->count(),
            'avg_time_to_hire' => $this->getAverageTimeToHire(),
            'conversion_rate' => $this->getConversionRate(),
        ];

        // Application Status Distribution
        $statusDistribution = Application::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();

        // Applications Over Time (Last 30 days)
        $applicationsOverTime = Application::select(
                DB::raw('DATE(applied_at) as date'),
                DB::raw('count(*) as count')
            )
            ->where('applied_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top Performing Job Postings
        $topJobPostings = JobPosting::withCount('applications')
            ->orderBy('applications_count', 'desc')
            ->limit(10)
            ->get();

        // Department Performance
        $departmentStats = DB::table('job_postings')
            ->join('departments', 'job_postings.department_id', '=', 'departments.id')
            ->leftJoin('applications', 'job_postings.id', '=', 'applications.job_posting_id')
            ->select(
                'departments.name as department_name',
                DB::raw('COUNT(DISTINCT job_postings.id) as job_postings_count'),
                DB::raw('COUNT(applications.id) as applications_count'),
                DB::raw('AVG(CASE WHEN applications.status = "offered" THEN 1 ELSE 0 END) * 100 as offer_rate')
            )
            ->groupBy('departments.id', 'departments.name')
            ->orderBy('applications_count', 'desc')
            ->get();

        // Recent Activity
        $recentApplications = Application::with(['jobPosting:id,title,department_id', 'jobPosting.department:id,name'])
            ->latest('applied_at')
            ->limit(10)
            ->get();

        return view('recruitment.reports.index', compact(
            'metrics',
            'statusDistribution',
            'applicationsOverTime',
            'topJobPostings',
            'departmentStats',
            'recentApplications',
            'dateRange'
        ));
    }

    /**
     * Generate detailed recruitment report
     */
    public function detailed(Request $request)
    {
        $startDate = $request->get('start_date', Carbon::now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', Carbon::now()->format('Y-m-d'));
        $department = $request->get('department');
        $jobPosting = $request->get('job_posting');

        $query = Application::with(['jobPosting.department', 'statusHistory.changedBy'])
            ->whereBetween('applied_at', [$startDate, $endDate]);

        if ($department) {
            $query->whereHas('jobPosting', function ($q) use ($department) {
                $q->where('department_id', $department);
            });
        }

        if ($jobPosting) {
            $query->where('job_posting_id', $jobPosting);
        }

        $applications = $query->orderBy('applied_at', 'desc')->paginate(50);

        // Summary statistics for the filtered data
        $baseQuery = Application::whereBetween('applied_at', [$startDate, $endDate]);

        if ($department) {
            $baseQuery->whereHas('jobPosting', function ($q) use ($department) {
                $q->where('department_id', $department);
            });
        }

        if ($jobPosting) {
            $baseQuery->where('job_posting_id', $jobPosting);
        }

        // Create separate query for status breakdown to avoid GROUP BY conflicts
        $statusQuery = Application::whereBetween('applied_at', [$startDate, $endDate]);
        
        if ($department) {
            $statusQuery->whereHas('jobPosting', function ($q) use ($department) {
                $q->where('department_id', $department);
            });
        }

        if ($jobPosting) {
            $statusQuery->where('job_posting_id', $jobPosting);
        }

        $summary = [
            'total_applications' => $baseQuery->count(),
            'status_breakdown' => $statusQuery->select('status', DB::raw('count(*) as count'))
                ->groupBy('status')
                ->pluck('count', 'status'),
            'avg_processing_time' => $this->getAverageProcessingTime($startDate, $endDate, $department, $jobPosting),
        ];

        $departments = \App\Models\Department::where('active', true)->get();
        $jobPostings = JobPosting::when($department, function ($q) use ($department) {
            return $q->where('department_id', $department);
        })->get();

        return view('recruitment.reports.detailed', compact(
            'applications',
            'summary',
            'departments',
            'jobPostings',
            'startDate',
            'endDate',
            'department',
            'jobPosting'
        ));
    }

    /**
     * Export recruitment data
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'csv');
        $startDate = $request->get('start_date', Carbon::now()->subDays(30)->format('Y-m-d'));
        $endDate = $request->get('end_date', Carbon::now()->format('Y-m-d'));

        $applications = Application::with(['jobPosting.department'])
            ->whereBetween('applied_at', [$startDate, $endDate])
            ->get();

        if ($format === 'csv') {
            return $this->exportToCsv($applications, $startDate, $endDate);
        }

        // Add other export formats as needed
        return redirect()->back()->with('error', 'Export format not supported');
    }

    /**
     * Get average time to hire in days
     */
    private function getAverageTimeToHire()
    {
        $hiredApplications = Application::where('status', 'offered')
            ->whereNotNull('applied_at')
            ->get();

        if ($hiredApplications->isEmpty()) {
            return 0;
        }

        $totalDays = $hiredApplications->sum(function ($application) {
            return $application->applied_at->diffInDays(now());
        });

        return round($totalDays / $hiredApplications->count(), 1);
    }

    /**
     * Get conversion rate (applications to offers)
     */
    private function getConversionRate()
    {
        $totalApplications = Application::count();
        $offeredApplications = Application::where('status', 'offered')->count();

        if ($totalApplications === 0) {
            return 0;
        }

        return round(($offeredApplications / $totalApplications) * 100, 1);
    }

    /**
     * Get average processing time for applications
     */
    private function getAverageProcessingTime($startDate, $endDate, $department = null, $jobPosting = null)
    {
        $query = Application::whereBetween('applied_at', [$startDate, $endDate])
            ->whereIn('status', ['offered', 'rejected']);

        if ($department) {
            $query->whereHas('jobPosting', function ($q) use ($department) {
                $q->where('department_id', $department);
            });
        }

        if ($jobPosting) {
            $query->where('job_posting_id', $jobPosting);
        }

        $applications = $query->get();

        if ($applications->isEmpty()) {
            return 0;
        }

        $totalDays = $applications->sum(function ($application) {
            return $application->applied_at->diffInDays($application->updated_at);
        });

        return round($totalDays / $applications->count(), 1);
    }

    /**
     * Export applications to CSV
     */
    private function exportToCsv($applications, $startDate, $endDate)
    {
        $filename = "recruitment_report_{$startDate}_to_{$endDate}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($applications) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'Application ID',
                'Applicant Name',
                'Email',
                'Phone',
                'Job Title',
                'Department',
                'Status',
                'Applied Date',
                'Last Updated'
            ]);

            // CSV Data
            foreach ($applications as $application) {
                fputcsv($file, [
                    $application->tracking_token,
                    $application->full_name,
                    $application->email,
                    $application->phone,
                    $application->jobPosting->title,
                    $application->jobPosting->department->name ?? 'N/A',
                    ucfirst($application->status),
                    $application->applied_at->format('Y-m-d H:i:s'),
                    $application->updated_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}