<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Attendance;
use App\Models\LeaveRequest;
use App\Models\Payroll;
use App\Models\Department;
use App\Models\TrainingProgram;
use App\Models\TrainingSession;
use App\Models\TrainingEnrollment;
use App\Models\TrainingCategory;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }

    public function attendance(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $departmentId = $request->input('department');

        $query = Attendance::with(['employee.department'])
            ->whereBetween('date', [$startDate, $endDate]);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $attendances = $query->orderBy('date', 'desc')->get();

        // Enhanced summary statistics
        $summary = [
            'total_days' => $attendances->count(),
            'present' => $attendances->where('status', 'present')->count(),
            'absent' => $attendances->where('status', 'absent')->count(),
            'late' => $attendances->where('status', 'late')->count(),
            'half_day' => $attendances->where('status', 'half_day')->count(),
            'on_leave' => $attendances->where('status', 'on_leave')->count(),
        ];

        // Calculate percentages for analytics
        $totalRecords = $summary['total_days'];
        $summary['present_percentage'] = $totalRecords > 0 ? round(($summary['present'] / $totalRecords) * 100, 1) : 0;
        $summary['absent_percentage'] = $totalRecords > 0 ? round(($summary['absent'] / $totalRecords) * 100, 1) : 0;
        $summary['late_percentage'] = $totalRecords > 0 ? round(($summary['late'] / $totalRecords) * 100, 1) : 0;
        $summary['attendance_rate'] = $totalRecords > 0 ? round((($summary['present'] + $summary['late']) / $totalRecords) * 100, 1) : 0;

        // Department-wise breakdown for analytics
        $departmentBreakdown = $attendances->groupBy('employee.department.name')
            ->map(function ($deptAttendances, $deptName) {
                $total = $deptAttendances->count();
                $present = $deptAttendances->where('status', 'present')->count();
                $late = $deptAttendances->where('status', 'late')->count();
                
                return [
                    'name' => $deptName ?: 'No Department',
                    'total' => $total,
                    'present' => $present,
                    'late' => $late,
                    'absent' => $deptAttendances->where('status', 'absent')->count(),
                    'attendance_rate' => $total > 0 ? round((($present + $late) / $total) * 100, 1) : 0
                ];
            })
            ->sortByDesc('total')
            ->take(10);

        $departments = Department::where('active', true)->get();

        return view('reports.attendance', compact(
            'attendances', 
            'summary', 
            'departments', 
            'startDate', 
            'endDate',
            'departmentBreakdown'
        ));
    }

    public function leave(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $departmentId = $request->input('department');

        $query = LeaveRequest::with(['employee.department', 'leaveType'])
            ->whereYear('start_date', $year);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $leaves = $query->orderBy('start_date', 'desc')->get();

        // Summary statistics
        $summary = [
            'total_requests' => $leaves->count(),
            'approved' => $leaves->where('status', 'approved')->count(),
            'pending' => $leaves->where('status', 'pending')->count(),
            'rejected' => $leaves->where('status', 'rejected')->count(),
            'total_days' => $leaves->where('status', 'approved')->sum('number_of_days'),
        ];

        $departments = Department::where('active', true)->get();
        $years = range(date('Y'), date('Y') - 5);

        return view('reports.leave', compact('leaves', 'summary', 'departments', 'years', 'year'));
    }

    public function payroll(Request $request)
    {
        $month = $request->input('month', date('m'));
        $year = $request->input('year', date('Y'));
        $departmentId = $request->input('department');

        $query = Payroll::with(['employee.department'])
            ->where('month', $month)
            ->where('year', $year);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $payrolls = $query->orderBy('employee_id')->get();

        // Enhanced summary statistics
        $summary = [
            'total_employees' => $payrolls->count(),
            'total_basic' => $payrolls->sum('basic_salary'),
            'total_allowances' => $payrolls->sum('total_allowances'),
            'total_deductions' => $payrolls->sum('total_deductions'),
            'total_gross' => $payrolls->sum('gross_salary'),
            'total_net' => $payrolls->sum('net_salary'),
            'paid' => $payrolls->where('status', 'paid')->count(),
            'pending' => $payrolls->where('status', 'pending')->count(),
        ];

        // Calculate additional analytics
        $summary['avg_gross'] = $summary['total_employees'] > 0 ? round($summary['total_gross'] / $summary['total_employees'], 0) : 0;
        $summary['avg_net'] = $summary['total_employees'] > 0 ? round($summary['total_net'] / $summary['total_employees'], 0) : 0;
        $summary['avg_deductions'] = $summary['total_employees'] > 0 ? round($summary['total_deductions'] / $summary['total_employees'], 0) : 0;
        $summary['payment_completion_rate'] = $summary['total_employees'] > 0 ? round(($summary['paid'] / $summary['total_employees']) * 100, 1) : 0;

        // Department-wise payroll breakdown
        $departmentBreakdown = $payrolls->groupBy('employee.department.name')
            ->map(function ($deptPayrolls, $deptName) {
                $count = $deptPayrolls->count();
                $totalNet = $deptPayrolls->sum('net_salary');
                $totalGross = $deptPayrolls->sum('gross_salary');
                
                return [
                    'name' => $deptName ?: 'No Department',
                    'employee_count' => $count,
                    'total_net' => $totalNet,
                    'total_gross' => $totalGross,
                    'avg_net' => $count > 0 ? round($totalNet / $count, 0) : 0,
                    'paid_count' => $deptPayrolls->where('status', 'paid')->count()
                ];
            })
            ->sortByDesc('total_net')
            ->take(10);

        $departments = Department::where('active', true)->get();
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];
        $years = range(date('Y'), date('Y') - 5);

        return view('reports.payroll', compact(
            'payrolls', 
            'summary', 
            'departments', 
            'months', 
            'years', 
            'month', 
            'year',
            'departmentBreakdown'
        ));
    }

    public function employee(Request $request)
    {
        $departmentId = $request->input('department');
        $status = $request->input('status', 'active');

        $query = Employee::with(['department', 'position']);

        if ($departmentId) {
            $query->where('department_id', $departmentId);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $employees = $query->orderBy('first_name')->get();

        // Summary statistics
        $summary = [
            'total_employees' => $employees->count(),
            'male' => $employees->where('gender', 'male')->count(),
            'female' => $employees->where('gender', 'female')->count(),
            'avg_salary' => round($employees->avg('salary'), 2),
        ];

        // Group by department
        $byDepartment = $employees->groupBy('department.name')->map(function($dept) {
            return $dept->count();
        });

        $departments = Department::where('active', true)->get();

        return view('reports.employee', compact('employees', 'summary', 'byDepartment', 'departments'));
    }

    public function exportAttendancePdf(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $departmentId = $request->input('department');

        $query = Attendance::with(['employee.department'])
            ->whereBetween('date', [$startDate, $endDate]);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $attendances = $query->orderBy('date', 'desc')->get();
        $departments = Department::where('active', true)->get();

        $pdf = Pdf::loadView('exports.attendance-pdf', compact('attendances', 'departments'));
        return $pdf->download('attendance-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportLeavePdf(Request $request)
    {
        $year = $request->input('year', date('Y'));
        $departmentId = $request->input('department');

        $query = LeaveRequest::with(['employee.department', 'leaveType'])
            ->whereYear('start_date', $year);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $leaves = $query->orderBy('start_date', 'desc')->get();
        $departments = Department::where('active', true)->get();

        $pdf = Pdf::loadView('exports.leave-pdf', compact('leaves', 'departments'));
        return $pdf->download('leave-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportPayrollPdf(Request $request)
    {
        $month = $request->input('month', date('m'));
        $year = $request->input('year', date('Y'));
        $departmentId = $request->input('department');

        $query = Payroll::with(['employee.department'])
            ->where('month', $month)
            ->where('year', $year);

        if ($departmentId) {
            $query->whereHas('employee', function($q) use ($departmentId) {
                $q->where('department_id', $departmentId);
            });
        }

        $payrolls = $query->orderBy('employee_id')->get();
        $departments = Department::where('active', true)->get();

        $pdf = Pdf::loadView('exports.payroll-pdf', compact('payrolls', 'departments'));
        return $pdf->download('payroll-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportEmployeePdf(Request $request)
    {
        $departmentId = $request->input('department');
        $status = $request->input('status', 'active');

        $query = Employee::with(['department', 'position']);

        if ($departmentId) {
            $query->where('department_id', $departmentId);
        }

        if ($status) {
            $query->where('status', $status);
        }

        $employees = $query->orderBy('first_name')->get();
        $departments = Department::where('active', true)->get();

        $pdf = Pdf::loadView('exports.employee-pdf', compact('employees', 'departments'));
        return $pdf->download('employee-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function training(Request $request)
    {
        // Check if user has permission to view training reports
        if (!auth()->user()->can('view_training_reports') && !auth()->user()->can('manage_training')) {
            abort(403, 'Unauthorized access to training reports.');
        }

        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $categoryId = $request->input('category');
        $departmentId = $request->input('department');
        $programId = $request->input('program');

        // Base query for training sessions within date range
        $sessionsQuery = TrainingSession::with(['program.category', 'enrollments.employee.department'])
            ->whereBetween('start_datetime', [$startDate, $endDate]);

        if ($categoryId) {
            $sessionsQuery->whereHas('program', function($q) use ($categoryId) {
                $q->where('training_category_id', $categoryId);
            });
        }

        if ($programId) {
            $sessionsQuery->where('training_program_id', $programId);
        }

        $sessions = $sessionsQuery->orderBy('start_datetime', 'desc')->get();

        // Filter enrollments by department if specified
        if ($departmentId) {
            $sessions = $sessions->map(function($session) use ($departmentId) {
                $session->enrollments = $session->enrollments->filter(function($enrollment) use ($departmentId) {
                    return $enrollment->employee->department_id == $departmentId;
                });
                return $session;
            });
        }

        // Calculate comprehensive statistics
        $allEnrollments = $sessions->flatMap->enrollments;
        
        $summary = [
            'total_sessions' => $sessions->count(),
            'completed_sessions' => $sessions->where('status', 'completed')->count(),
            'total_enrollments' => $allEnrollments->count(),
            'completed_enrollments' => $allEnrollments->where('status', 'completed')->count(),
            'attended_enrollments' => $allEnrollments->whereIn('status', ['attended', 'completed'])->count(),
            'no_show_enrollments' => $allEnrollments->where('status', 'no_show')->count(),
            'total_training_hours' => $sessions->sum(function($session) {
                return $session->duration_in_hours * $session->enrollments->whereIn('status', ['attended', 'completed'])->count();
            }),
            'unique_participants' => $allEnrollments->pluck('employee_id')->unique()->count(),
        ];

        // Calculate rates
        $summary['completion_rate'] = $summary['total_enrollments'] > 0 
            ? round(($summary['completed_enrollments'] / $summary['total_enrollments']) * 100, 1) 
            : 0;
        
        $summary['attendance_rate'] = $summary['total_enrollments'] > 0 
            ? round(($summary['attended_enrollments'] / $summary['total_enrollments']) * 100, 1) 
            : 0;

        $summary['no_show_rate'] = $summary['total_enrollments'] > 0 
            ? round(($summary['no_show_enrollments'] / $summary['total_enrollments']) * 100, 1) 
            : 0;

        // Category-wise breakdown
        $categoryBreakdown = $allEnrollments->groupBy('session.program.category.name')
            ->map(function ($categoryEnrollments, $categoryName) {
                $total = $categoryEnrollments->count();
                $completed = $categoryEnrollments->where('status', 'completed')->count();
                $attended = $categoryEnrollments->whereIn('status', ['attended', 'completed'])->count();
                
                return [
                    'name' => $categoryName ?: 'No Category',
                    'total_enrollments' => $total,
                    'completed' => $completed,
                    'attended' => $attended,
                    'completion_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                    'attendance_rate' => $total > 0 ? round(($attended / $total) * 100, 1) : 0,
                ];
            })
            ->sortByDesc('total_enrollments');

        // Department-wise breakdown
        $departmentBreakdown = $allEnrollments->groupBy('employee.department.name')
            ->map(function ($deptEnrollments, $deptName) {
                $total = $deptEnrollments->count();
                $completed = $deptEnrollments->where('status', 'completed')->count();
                $attended = $deptEnrollments->whereIn('status', ['attended', 'completed'])->count();
                $uniqueEmployees = $deptEnrollments->pluck('employee_id')->unique()->count();
                
                return [
                    'name' => $deptName ?: 'No Department',
                    'total_enrollments' => $total,
                    'unique_employees' => $uniqueEmployees,
                    'completed' => $completed,
                    'attended' => $attended,
                    'completion_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                    'attendance_rate' => $total > 0 ? round(($attended / $total) * 100, 1) : 0,
                ];
            })
            ->sortByDesc('total_enrollments');

        // Top performing programs
        $programPerformance = $sessions->groupBy('program.title')
            ->map(function ($programSessions, $programTitle) {
                $allProgramEnrollments = $programSessions->flatMap->enrollments;
                $total = $allProgramEnrollments->count();
                $completed = $allProgramEnrollments->where('status', 'completed')->count();
                $attended = $allProgramEnrollments->whereIn('status', ['attended', 'completed'])->count();
                
                return [
                    'title' => $programTitle,
                    'sessions_count' => $programSessions->count(),
                    'total_enrollments' => $total,
                    'completed' => $completed,
                    'attended' => $attended,
                    'completion_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                    'attendance_rate' => $total > 0 ? round(($attended / $total) * 100, 1) : 0,
                ];
            })
            ->sortByDesc('completion_rate')
            ->take(10);

        // Cost analysis
        $costAnalysis = [
            'total_program_cost' => $sessions->sum(function($session) {
                return $session->program->cost_per_participant * $session->enrollments->whereIn('status', ['attended', 'completed'])->count();
            }),
            'total_actual_cost' => $sessions->sum('actual_cost'),
            'avg_cost_per_participant' => 0,
            'avg_cost_per_hour' => 0,
        ];

        if ($summary['attended_enrollments'] > 0) {
            $costAnalysis['avg_cost_per_participant'] = round(
                ($costAnalysis['total_program_cost'] + $costAnalysis['total_actual_cost']) / $summary['attended_enrollments'], 
                2
            );
        }

        if ($summary['total_training_hours'] > 0) {
            $costAnalysis['avg_cost_per_hour'] = round(
                ($costAnalysis['total_program_cost'] + $costAnalysis['total_actual_cost']) / $summary['total_training_hours'], 
                2
            );
        }

        // Get filter options
        $categories = TrainingCategory::active()->get();
        $departments = Department::where('active', true)->get();
        $programs = TrainingProgram::active()->get();

        return view('reports.training', compact(
            'sessions',
            'summary',
            'categoryBreakdown',
            'departmentBreakdown',
            'programPerformance',
            'costAnalysis',
            'categories',
            'departments',
            'programs',
            'startDate',
            'endDate'
        ));
    }

    public function exportTrainingPdf(Request $request)
    {
        // Check if user has permission to view training reports
        if (!auth()->user()->can('view_training_reports') && !auth()->user()->can('manage_training')) {
            abort(403, 'Unauthorized access to training reports.');
        }

        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->endOfMonth()->format('Y-m-d'));
        $categoryId = $request->input('category');
        $departmentId = $request->input('department');
        $programId = $request->input('program');

        // Use same logic as training report method
        $sessionsQuery = TrainingSession::with(['program.category', 'enrollments.employee.department'])
            ->whereBetween('start_datetime', [$startDate, $endDate]);

        if ($categoryId) {
            $sessionsQuery->whereHas('program', function($q) use ($categoryId) {
                $q->where('training_category_id', $categoryId);
            });
        }

        if ($programId) {
            $sessionsQuery->where('training_program_id', $programId);
        }

        $sessions = $sessionsQuery->orderBy('start_datetime', 'desc')->get();

        if ($departmentId) {
            $sessions = $sessions->map(function($session) use ($departmentId) {
                $session->enrollments = $session->enrollments->filter(function($enrollment) use ($departmentId) {
                    return $enrollment->employee->department_id == $departmentId;
                });
                return $session;
            });
        }

        $allEnrollments = $sessions->flatMap->enrollments;
        
        $summary = [
            'total_sessions' => $sessions->count(),
            'completed_sessions' => $sessions->where('status', 'completed')->count(),
            'total_enrollments' => $allEnrollments->count(),
            'completed_enrollments' => $allEnrollments->where('status', 'completed')->count(),
            'attended_enrollments' => $allEnrollments->whereIn('status', ['attended', 'completed'])->count(),
            'completion_rate' => $allEnrollments->count() > 0 
                ? round(($allEnrollments->where('status', 'completed')->count() / $allEnrollments->count()) * 100, 1) 
                : 0,
        ];

        $pdf = Pdf::loadView('exports.training-pdf', compact('sessions', 'summary', 'startDate', 'endDate'));
        return $pdf->download('training-report-' . now()->format('Y-m-d') . '.pdf');
    }
}
