<?php

namespace App\Http\Controllers;

use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RoleController extends Controller
{
    public function index()
    {
        $this->authorize('viewAny', Role::class);
        
        $query = Role::withCount('users');
        
        // Org admins cannot see system admin roles
        if (!Auth::user()->isSystemAdmin()) {
            $query->whereNotIn('name', ['system_admin']);
        }
        
        $roles = $query->get();
        return view('roles.index', compact('roles'));
    }

    public function create()
    {
        $this->authorize('create', Role::class);
        
        $permissions = $this->getAvailablePermissions();
        
        return view('roles.create', compact('permissions'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Role::class);
        
        $validated = $request->validate([
            'name' => 'required|string|unique:roles|max:255',
            'permissions' => 'array',
        ]);

        $role = Role::create([
            'name' => $validated['name'],
            'guard_name' => 'web',
        ]);

        if ($request->has('permissions')) {
            // Filter permissions to only those available to the current user
            $availablePermissions = $this->getAvailablePermissions()->flatten()->pluck('id')->toArray();
            $requestedPermissionIds = array_intersect($request->permissions, $availablePermissions);
            
            // Convert IDs to Permission models
            $permissions = Permission::whereIn('id', $requestedPermissionIds)->get();
            $role->givePermissionTo($permissions);
        }

        return redirect()->route('roles.index')->with('success', 'Role created successfully!');
    }

    public function show(Role $role)
    {
        $this->authorize('view', $role);
        
        $role->load(['permissions', 'users']);
        
        return view('roles.show', compact('role'));
    }

    public function edit(Role $role)
    {
        $this->authorize('update', $role);
        
        $permissions = $this->getAvailablePermissions();
        
        $rolePermissions = $role->permissions->pluck('id')->toArray();

        return view('roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    public function update(Request $request, Role $role)
    {
        $this->authorize('update', $role);
        
        $validated = $request->validate([
            'permissions' => 'array',
        ]);

        if ($request->has('permissions')) {
            // Filter permissions to only those available to the current user
            $availablePermissions = $this->getAvailablePermissions()->flatten()->pluck('id')->toArray();
            $requestedPermissionIds = array_intersect($request->permissions, $availablePermissions);
            
            // Convert IDs to Permission models
            $permissions = Permission::whereIn('id', $requestedPermissionIds)->get();
            $role->syncPermissions($permissions);
        } else {
            $role->syncPermissions([]);
        }

        return redirect()->route('roles.index')->with('success', 'Role updated successfully!');
    }

    public function destroy(Role $role)
    {
        $this->authorize('delete', $role);
        
        // Check if role has users
        if ($role->users()->count() > 0) {
            return redirect()->route('roles.index')->with('error', 'Cannot delete role with assigned users.');
        }

        $role->delete();

        return redirect()->route('roles.index')->with('success', 'Role deleted successfully!');
    }

    /**
     * Get permissions available to the current user
     */
    private function getAvailablePermissions()
    {
        $query = Permission::query();
        
        // System admins can see all permissions
        if (!Auth::user()->isSystemAdmin()) {
            // Org admins cannot see system admin permissions
            $query->whereNotIn('name', [
                'manage_system_admin',
                'view_system_admin',
                'create_system_admin',
                'edit_system_admin',
                'delete_system_admin'
            ]);
        }
        
        return $query->get()->groupBy(function($permission) {
            // Group permissions by prefix (e.g., 'view_employees' -> 'employees')
            $parts = explode('_', $permission->name);
            return count($parts) > 1 ? $parts[1] : 'general';
        });
    }
}
