<?php

namespace App\Http\Controllers;

use App\Services\SecurityMonitoringService;
use Illuminate\Http\Request;

class SecurityDashboardController extends Controller
{
    protected $securityService;

    public function __construct(SecurityMonitoringService $securityService)
    {
        $this->securityService = $securityService;
        $this->middleware(['auth', 'permission:view_security_dashboard']);
    }

    /**
     * Display security dashboard
     */
    public function index()
    {
        $report = $this->securityService->generateSecurityReport(7);
        $alerts = $this->securityService->getRecentAlerts(20);

        return view('security.dashboard', compact('report', 'alerts'));
    }

    /**
     * Display security logs
     */
    public function logs(Request $request)
    {
        $days = $request->get('days', 7);
        $type = $request->get('type', 'all');

        // In a real implementation, you'd query actual log data
        $logs = collect([
            [
                'id' => 1,
                'event' => 'failed_login_attempt',
                'level' => 'warning',
                'message' => 'Failed login attempt',
                'data' => ['email' => 'user@example.com', 'ip' => '192.168.1.1'],
                'created_at' => now()->subHours(2),
            ],
            [
                'id' => 2,
                'event' => 'suspicious_input_detected',
                'level' => 'error',
                'message' => 'Suspicious input patterns detected',
                'data' => ['url' => '/dashboard', 'inputs' => ['search']],
                'created_at' => now()->subHours(5),
            ],
            [
                'id' => 3,
                'event' => 'new_device_access',
                'level' => 'info',
                'message' => 'New device access detected',
                'data' => ['user_id' => 1, 'new_agent' => 'Mozilla/5.0...'],
                'created_at' => now()->subDay(),
            ],
        ]);

        if ($type !== 'all') {
            $logs = $logs->where('event', $type);
        }

        return view('security.logs', compact('logs', 'days', 'type'));
    }

    /**
     * Display IP whitelist management
     */
    public function ipWhitelist()
    {
        $whitelist = config('security.ip_whitelist', []);
        
        return view('security.ip-whitelist', compact('whitelist'));
    }

    /**
     * Add IP to whitelist
     */
    public function addIpToWhitelist(Request $request)
    {
        $request->validate([
            'ip_address' => 'required|string',
            'description' => 'nullable|string|max:255',
        ]);

        // In a real implementation, you'd store this in database or config
        $this->securityService->logSecurityEvent('ip_whitelist_added', [
            'ip_address' => $request->ip_address,
            'description' => $request->description,
            'added_by' => auth()->id(),
        ], 'info');

        return back()->with('success', 'IP address added to whitelist successfully.');
    }

    /**
     * Remove IP from whitelist
     */
    public function removeIpFromWhitelist(Request $request)
    {
        $request->validate([
            'ip_address' => 'required|string',
        ]);

        // In a real implementation, you'd remove this from database or config
        $this->securityService->logSecurityEvent('ip_whitelist_removed', [
            'ip_address' => $request->ip_address,
            'removed_by' => auth()->id(),
        ], 'info');

        return back()->with('success', 'IP address removed from whitelist successfully.');
    }

    /**
     * Display security settings
     */
    public function settings()
    {
        $settings = [
            'max_login_attempts' => config('security.max_login_attempts', 5),
            'lockout_duration' => config('security.lockout_duration', 15),
            'session_timeout' => config('security.session_timeout', 120),
            'require_2fa' => config('security.require_2fa', false),
            'password_expiry_days' => config('security.password_expiry_days', 90),
        ];

        return view('security.settings', compact('settings'));
    }

    /**
     * Update security settings
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'max_login_attempts' => 'required|integer|min:3|max:10',
            'lockout_duration' => 'required|integer|min:5|max:60',
            'session_timeout' => 'required|integer|min:30|max:480',
            'require_2fa' => 'boolean',
            'password_expiry_days' => 'required|integer|min:30|max:365',
        ]);

        // In a real implementation, you'd update the config or database
        $this->securityService->logSecurityEvent('security_settings_updated', [
            'settings' => $request->only([
                'max_login_attempts',
                'lockout_duration', 
                'session_timeout',
                'require_2fa',
                'password_expiry_days'
            ]),
            'updated_by' => auth()->id(),
        ], 'info');

        return back()->with('success', 'Security settings updated successfully.');
    }
}