<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use App\Http\Requests\UpdateThemeRequest;
use App\Models\ThemeSetting;
use App\Services\LogoUploadService;
use App\Services\ThemeService;
use App\Jobs\ProcessLogoUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ThemeController extends Controller
{
    public function __construct(
        protected LogoUploadService $logoService,
        protected ThemeService $themeService
    ) {
        $this->middleware(['auth', 'role:org_admin']);
    }

    /**
     * Display theme management page
     */
    public function index()
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            abort(403, 'You must belong to an organization to manage themes.');
        }
        
        $themeSetting = ThemeSetting::getForOrganization($organization->id);
        $presetThemes = config('theme.themes');
        $currentTheme = $themeSetting?->base_theme ?? 'default';
        
        return view('settings.theme.index', compact(
            'organization',
            'themeSetting',
            'presetThemes',
            'currentTheme'
        ));
    }

    /**
     * Update theme settings
     */
    public function update(UpdateThemeRequest $request)
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return redirect()->back()->with('error', 'You must belong to an organization.');
        }
        
        $data = $request->validated();
        
        // Handle logo uploads (queue for background processing)
        foreach (['logo', 'sidebar_logo', 'header_logo', 'favicon', 'login_banner'] as $logoType) {
            if ($request->hasFile($logoType)) {
                try {
                    // Store temp file
                    $file = $request->file($logoType);
                    $tempPath = $file->store('temp/logos', 'local');
                    
                    // Queue processing job
                    ProcessLogoUpload::dispatch(
                        $tempPath,
                        $logoType,
                        $organization->id,
                        $file->getClientOriginalExtension()
                    );
                    
                } catch (\Exception $e) {
                    return redirect()
                        ->back()
                        ->with('error', "Failed to queue {$logoType} upload: " . $e->getMessage())
                        ->withInput();
                }
            }
        }
        
        // Remove file inputs from data
        unset($data['logo'], $data['sidebar_logo'], $data['header_logo'], $data['favicon'], $data['login_banner']);
        
        // Update or create theme settings
        $themeSetting = ThemeSetting::updateOrCreate(
            ['organization_id' => $organization->id],
            array_merge($data, ['is_active' => true])
        );
        
        // Generate static CSS file
        $this->generateStaticThemeCSS($organization->id);
        
        // Clear theme cache
        $this->themeService->clearCache($organization->id);
        
        $message = 'Theme updated successfully!';
        if ($request->hasFile(['logo', 'sidebar_logo', 'header_logo', 'favicon', 'login_banner'])) {
            $message .= ' Logo uploads are being processed in the background.';
        }
        
        return redirect()
            ->route('settings.theme.index')
            ->with('success', $message . ' Refresh the page to see changes.');
    }

    /**
     * Reset theme to base theme (remove all customizations)
     */
    public function reset()
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return redirect()->back()->with('error', 'You must belong to an organization.');
        }
        
        $themeSetting = ThemeSetting::where('organization_id', $organization->id)->first();
        
        if ($themeSetting) {
            // Keep base_theme and logos, reset everything else
            $themeSetting->update([
                'primary_color' => null,
                'secondary_color' => null,
                'accent_color' => null,
                'success_color' => null,
                'warning_color' => null,
                'danger_color' => null,
                'info_color' => null,
                'light_color' => null,
                'dark_color' => null,
                'sidebar_bg_color' => null,
                'sidebar_text_color' => null,
                'sidebar_hover_color' => null,
                'sidebar_active_color' => null,
                'primary_font' => null,
                'heading_font' => null,
                'custom_css' => null,
                'sidebar_collapsed_default' => false,
                'dark_mode_enabled' => false,
            ]);
            
            // Regenerate static CSS
            $this->generateStaticThemeCSS($organization->id);
            
            // Clear cache
            $this->themeService->clearCache($organization->id);
        }
        
        return redirect()
            ->route('settings.theme.index')
            ->with('success', 'Theme reset to base theme! Refresh the page to see changes.');
    }

    /**
     * Upload logo via AJAX
     */
    public function uploadLogo(Request $request)
    {
        $request->validate([
            'logo' => 'required|image|max:2048',
            'type' => 'required|in:logo,sidebar_logo,header_logo,favicon,login_banner',
        ]);
        
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return response()->json(['error' => 'No organization found'], 403);
        }
        
        try {
            // Store temp file
            $file = $request->file('logo');
            $tempPath = $file->store('temp/logos', 'local');
            
            // Queue processing job
            ProcessLogoUpload::dispatch(
                $tempPath,
                $request->type,
                $organization->id,
                $file->getClientOriginalExtension()
            );
            
            return response()->json([
                'success' => true,
                'message' => 'Logo upload queued for processing',
            ]);
            
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Delete logo
     */
    public function deleteLogo($type)
    {
        $organization = auth()->user()->organization;
        
        if (!$organization) {
            return response()->json(['error' => 'No organization found'], 403);
        }
        
        $themeSetting = ThemeSetting::where('organization_id', $organization->id)->first();
        
        if ($themeSetting) {
            $this->logoService->delete($themeSetting->{"{$type}_url"});
            $themeSetting->update(["{$type}_url" => null]);
            
            // Regenerate static CSS
            $this->generateStaticThemeCSS($organization->id);
            
            // Clear cache
            $this->themeService->clearCache($organization->id);
        }
        
        return response()->json(['success' => true]);
    }

    /**
     * Generate static CSS file for organization theme
     */
    protected function generateStaticThemeCSS(int $organizationId): void
    {
        try {
            $css = $this->themeService->generateStaticCSS($organizationId);
            
            if ($css) {
                // Ensure directory exists
                Storage::disk('public')->makeDirectory('themes');
                
                // Save CSS file
                Storage::disk('public')->put(
                    "themes/org_{$organizationId}.css",
                    $css
                );
            }
        } catch (\Exception $e) {
            // Log error but don't fail the request
            \Log::error("Failed to generate static theme CSS: " . $e->getMessage());
        }
    }
}

