<?php

namespace App\Http\Controllers\SystemAdmin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Organization;
use App\Services\AuditService;
use App\Exports\AuditLogsExport;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class AuditLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('system_admin'); // Ensure only system admins can access
        $this->middleware('permission:view_system_audit_logs');
    }

    /**
     * System Admin Audit Dashboard - can see all organizations
     */
    public function index(Request $request): View|JsonResponse
    {
        $query = AuditLog::with(['user', 'organization'])
            ->latest();

        // Apply filters
        $this->applyFilters($query, $request);

        // Handle AJAX requests for real-time filtering
        if ($request->ajax()) {
            $logs = $query->paginate(25);
            return response()->json([
                'html' => view('systemadmin.audit-logs.table', compact('logs'))->render(),
                'pagination' => $logs->links()->render(),
                'total' => $logs->total(),
            ]);
        }

        $logs = $query->paginate(25);
        
        // Get statistics for all organizations
        $statistics = AuditService::getStatistics(null, 30);
        
        // Get organizations for filter dropdown
        $organizations = Organization::select('id', 'name')->get();

        return view('systemadmin.audit-logs.index', compact('logs', 'statistics', 'organizations'));
    }

    /**
     * System Admin Dashboard - Overview of all audit activities
     */
    public function dashboard(Request $request): View
    {
        $days = $request->get('days', 30);
        
        // Get overall statistics
        $overallStats = AuditService::getStatistics(null, $days);
        
        // Get per-organization statistics
        $organizations = Organization::with(['users'])->get();
        $orgStats = [];
        
        foreach ($organizations as $org) {
            $orgStats[] = [
                'organization' => $org,
                'stats' => AuditService::getStatistics($org->id, $days)
            ];
        }

        // Get security alerts (high/critical severity events)
        $securityAlerts = AuditLog::with(['user', 'organization'])
            ->whereIn('severity', ['high', 'critical'])
            ->where('created_at', '>=', now()->subDays($days))
            ->latest()
            ->limit(10)
            ->get();

        // Get top active organizations
        $topOrganizations = AuditLog::selectRaw('organization_id, COUNT(*) as activity_count')
            ->whereNotNull('organization_id')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('organization_id')
            ->orderByDesc('activity_count')
            ->limit(5)
            ->with('organization')
            ->get();

        return view('systemadmin.audit-logs.dashboard', compact(
            'overallStats', 
            'orgStats', 
            'securityAlerts', 
            'topOrganizations',
            'days'
        ));
    }

    /**
     * Show detailed audit log entry
     */
    public function show(AuditLog $auditLog): View
    {
        // System admin can view any audit log
        return view('systemadmin.audit-logs.show', compact('auditLog'));
    }

    /**
     * Export audit logs with system admin privileges
     */
    public function export(Request $request)
    {
        $request->validate([
            'format' => 'required|in:csv,xlsx',
            'from_date' => 'nullable|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'organization_id' => 'nullable|exists:organizations,id',
        ]);

        $query = AuditLog::with(['user', 'organization']);

        // System admin can export from specific organization or all
        if ($request->organization_id) {
            $query->forOrganization($request->organization_id);
        }

        // Apply date range filter
        if ($request->from_date && $request->to_date) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        $this->applyFilters($query, $request);

        // Limit export size for performance
        $maxRecords = config('audit.export.max_records', 50000);
        if ((clone $query)->count() > $maxRecords) {
            return back()->withErrors([
                'export' => "Export limited to {$maxRecords} records. Please narrow your filters."
            ]);
        }

        $orgName = $request->organization_id 
            ? '_' . Organization::find($request->organization_id)->name 
            : '_all_orgs';
            
        $filename = 'system_audit_logs' . $orgName . '_' . now()->format('Y-m-d_H-i-s') . '.' . $request->format;

        return Excel::download(
            new AuditLogsExport($query),
            $filename
        );
    }

    /**
     * Get organization comparison statistics
     */
    public function organizationComparison(Request $request): JsonResponse
    {
        $days = $request->get('days', 30);
        
        $organizations = Organization::with(['users'])->get();
        $comparison = [];
        
        foreach ($organizations as $org) {
            $stats = AuditService::getStatistics($org->id, $days);
            $comparison[] = [
                'organization_name' => $org->name,
                'total_activities' => $stats['total_activities'],
                'user_count' => $org->users->count(),
                'activities_per_user' => $org->users->count() > 0 
                    ? round($stats['total_activities'] / $org->users->count(), 2) 
                    : 0,
                'security_events' => AuditLog::forOrganization($org->id)
                    ->where('category', 'security')
                    ->where('created_at', '>=', now()->subDays($days))
                    ->count(),
            ];
        }

        return response()->json($comparison);
    }

    /**
     * Apply filters to the audit log query
     */
    private function applyFilters($query, Request $request): void
    {
        if ($request->filled('organization_id')) {
            $query->forOrganization($request->organization_id);
        }

        if ($request->filled('user_id')) {
            $query->byUser($request->user_id);
        }

        if ($request->filled('event_type')) {
            $query->byEventType($request->event_type);
        }

        if ($request->filled('category')) {
            $query->byCategory($request->category);
        }

        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->dateRange($request->from_date, $request->to_date);
        }

        if ($request->filled('search')) {
            $query->search($request->search);
        }

        if ($request->filled('auditable_type')) {
            $query->where('auditable_type', 'like', '%' . $request->auditable_type . '%');
        }
    }
}