<?php

namespace App\Http\Controllers\SystemAdmin;

use App\Http\Controllers\Controller;
use App\Models\Organization;
use App\Models\OrganizationSubscription;
use App\Models\SubscriptionPlan;
use App\Services\SubscriptionService;
use Illuminate\Http\Request;

class SubscriptionManagementController extends Controller
{
    protected $subscriptionService;

    public function __construct(SubscriptionService $subscriptionService)
    {
        $this->middleware('system_admin');
        $this->subscriptionService = $subscriptionService;
    }

    /**
     * Display all subscriptions
     */
    public function index(Request $request)
    {
        $query = OrganizationSubscription::with(['organization', 'plan']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by plan
        if ($request->filled('plan_id')) {
            $query->where('subscription_plan_id', $request->plan_id);
        }

        // Search by organization name
        if ($request->filled('search')) {
            $query->whereHas('organization', function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%');
            });
        }

        $subscriptions = $query->latest()->paginate(20);
        $plans = SubscriptionPlan::active()->get();

        // Statistics
        $stats = [
            'total' => OrganizationSubscription::count(),
            'active' => OrganizationSubscription::where('status', 'active')->count(),
            'trial' => OrganizationSubscription::where('status', 'trial')->count(),
            'suspended' => OrganizationSubscription::where('status', 'suspended')->count(),
            'cancelled' => OrganizationSubscription::where('status', 'cancelled')->count(),
        ];

        return view('systemadmin.subscriptions.index', compact('subscriptions', 'plans', 'stats'));
    }

    /**
     * Show subscription details
     */
    public function show(OrganizationSubscription $subscription)
    {
        $subscription->load(['organization', 'plan']);
        $summary = $this->subscriptionService->getSubscriptionSummary($subscription->organization);

        return view('systemadmin.subscriptions.show', compact('subscription', 'summary'));
    }

    /**
     * Show edit form
     */
    public function edit(OrganizationSubscription $subscription)
    {
        $plans = SubscriptionPlan::active()->get();
        return view('systemadmin.subscriptions.edit', compact('subscription', 'plans'));
    }

    /**
     * Update subscription
     */
    public function update(Request $request, OrganizationSubscription $subscription)
    {
        $request->validate([
            'subscription_plan_id' => 'required|exists:subscription_plans,id',
            'billing_cycle' => 'required|in:monthly,annual',
            'status' => 'required|in:trial,active,suspended,cancelled',
        ]);

        try {
            // Change plan if different
            if ($request->subscription_plan_id != $subscription->subscription_plan_id) {
                $newPlan = SubscriptionPlan::find($request->subscription_plan_id);
                $this->subscriptionService->changePlan($subscription, $newPlan, true);
            }

            // Update other fields
            $subscription->update([
                'billing_cycle' => $request->billing_cycle,
                'status' => $request->status,
            ]);

            return redirect()
                ->route('systemadmin.subscriptions.show', $subscription)
                ->with('success', 'Subscription updated successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update subscription: ' . $e->getMessage());
        }
    }

    /**
     * Extend trial
     */
    public function extendTrial(Request $request, OrganizationSubscription $subscription)
    {
        $request->validate([
            'days' => 'required|integer|min:1|max:30',
        ]);

        try {
            $this->subscriptionService->extendTrial($subscription, $request->days);

            return back()->with('success', "Trial extended by {$request->days} days");

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to extend trial: ' . $e->getMessage());
        }
    }

    /**
     * Suspend subscription
     */
    public function suspend(Request $request, OrganizationSubscription $subscription)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        try {
            $this->subscriptionService->suspendSubscription($subscription, $request->reason);

            return back()->with('success', 'Subscription suspended');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to suspend subscription: ' . $e->getMessage());
        }
    }

    /**
     * Reactivate subscription
     */
    public function reactivate(OrganizationSubscription $subscription)
    {
        try {
            $this->subscriptionService->reactivateSubscription($subscription);

            return back()->with('success', 'Subscription reactivated');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to reactivate subscription: ' . $e->getMessage());
        }
    }

    /**
     * Cancel subscription
     */
    public function cancel(Request $request, OrganizationSubscription $subscription)
    {
        $request->validate([
            'reason' => 'nullable|string|max:500',
            'immediately' => 'boolean',
        ]);

        try {
            $this->subscriptionService->cancelSubscription(
                $subscription,
                $request->reason,
                $request->boolean('immediately')
            );

            $message = $request->boolean('immediately') ? 
                'Subscription cancelled immediately' : 
                'Subscription will be cancelled at end of billing period';

            return back()->with('success', $message);

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to cancel subscription: ' . $e->getMessage());
        }
    }
}
