<?php

namespace App\Http\Controllers;

use App\Models\TrainingRecommendation;
use App\Models\TrainingProgram;
use App\Models\Employee;
use App\Models\Skill;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TrainingRecommendationController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', TrainingRecommendation::class);

        $query = TrainingRecommendation::with(['employee', 'trainingProgram', 'recommendedBy']);

        // Filter by employee (for managers viewing their team)
        if ($request->filled('employee')) {
            $query->where('employee_id', $request->employee);
        } elseif (auth()->user()->employee && !auth()->user()->can('view_all_training_recommendations')) {
            // Regular employees see only their recommendations
            $query->where('employee_id', auth()->user()->employee->id);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('recommendation_type', $request->type);
        }

        // Filter by priority
        if ($request->filled('priority')) {
            switch ($request->priority) {
                case 'high':
                    $query->where('priority_score', '>=', 80);
                    break;
                case 'medium':
                    $query->whereBetween('priority_score', [50, 79]);
                    break;
                case 'low':
                    $query->where('priority_score', '<', 50);
                    break;
            }
        }

        $recommendations = $query->orderBy('priority_score', 'desc')
            ->orderBy('recommended_at', 'desc')
            ->paginate(15);

        // Statistics
        $stats = [
            'total' => TrainingRecommendation::count(),
            'pending' => TrainingRecommendation::where('status', 'pending')->count(),
            'accepted' => TrainingRecommendation::where('status', 'accepted')->count(),
            'completed' => TrainingRecommendation::where('status', 'completed')->count(),
            'high_priority' => TrainingRecommendation::where('priority_score', '>=', 80)->count(),
        ];

        $employees = Employee::where('status', 'active')->get();

        return view('training.recommendations.index', compact('recommendations', 'stats', 'employees'));
    }

    public function create(Request $request)
    {
        $this->authorize('create', TrainingRecommendation::class);

        $employees = Employee::where('status', 'active')->get();
        $trainingPrograms = TrainingProgram::where('is_active', true)->with('category')->get();

        // Pre-select employee if provided
        $selectedEmployee = null;
        if ($request->filled('employee_id')) {
            $selectedEmployee = Employee::find($request->employee_id);
        }

        // Pre-select program if provided
        $selectedProgram = null;
        if ($request->filled('training_program_id')) {
            $selectedProgram = TrainingProgram::find($request->training_program_id);
        }

        return view('training.recommendations.create', compact('employees', 'trainingPrograms', 'selectedEmployee', 'selectedProgram'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', TrainingRecommendation::class);

        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'training_program_id' => 'required|exists:training_programs,id',
            'recommendation_type' => 'required|in:skill_gap,career_development,performance_improvement,mandatory,peer_suggestion',
            'reason' => 'required|string|max:1000',
            'priority_score' => 'required|integer|min:1|max:100',
            'supporting_data' => 'nullable|array',
        ]);

        $validated['status'] = 'pending';
        $validated['recommended_at'] = now();
        $validated['recommended_by'] = auth()->id();

        $recommendation = TrainingRecommendation::create($validated);

        return redirect()->route('training.recommendations.index')
            ->with('success', 'Training recommendation created successfully!');
    }

    public function show(TrainingRecommendation $recommendation)
    {
        $this->authorize('view', $recommendation);

        $recommendation->load(['employee', 'trainingProgram.category', 'recommendedBy']);

        return view('training.recommendations.show', compact('recommendation'));
    }

    public function respond(Request $request, TrainingRecommendation $recommendation)
    {
        $this->authorize('respond', $recommendation);

        $validated = $request->validate([
            'response' => 'required|in:accept,decline',
            'response_notes' => 'nullable|string|max:500',
        ]);

        if ($validated['response'] === 'accept') {
            $recommendation->accept();
            $message = 'Training recommendation accepted successfully!';
        } else {
            $recommendation->decline();
            $message = 'Training recommendation declined.';
        }

        // Store response notes if provided
        if (!empty($validated['response_notes'])) {
            $supportingData = $recommendation->supporting_data ?? [];
            $supportingData['response_notes'] = $validated['response_notes'];
            $recommendation->update(['supporting_data' => $supportingData]);
        }

        return back()->with('success', $message);
    }

    public function generateRecommendations(Request $request)
    {
        $this->authorize('create', TrainingRecommendation::class);

        $validated = $request->validate([
            'employee_id' => 'nullable|exists:employees,id',
            'recommendation_type' => 'required|in:skill_gap,performance_improvement,career_development',
        ]);

        $employees = isset($validated['employee_id']) && $validated['employee_id']
            ? [Employee::find($validated['employee_id'])]
            : Employee::where('status', 'active')->get();

        $generatedCount = 0;

        foreach ($employees as $employee) {
            $recommendations = $this->generateEmployeeRecommendations($employee, $validated['recommendation_type']);
            $generatedCount += count($recommendations);
        }

        return back()->with('success', "Generated {$generatedCount} training recommendations!");
    }

    private function generateEmployeeRecommendations(Employee $employee, string $type)
    {
        $recommendations = [];

        switch ($type) {
            case 'skill_gap':
                $recommendations = $this->generateSkillGapRecommendations($employee);
                break;
            case 'performance_improvement':
                $recommendations = $this->generatePerformanceRecommendations($employee);
                break;
            case 'career_development':
                $recommendations = $this->generateCareerRecommendations($employee);
                break;
        }

        return $recommendations;
    }

    private function generateSkillGapRecommendations(Employee $employee)
    {
        $recommendations = [];
        
        // Get employee's current skills and identify gaps
        $employeeSkills = $employee->skills()->get()->pluck('pivot.proficiency_level', 'id')->toArray();
        
        // Check if position exists and has skills relationship
        if (!$employee->position || !method_exists($employee->position, 'skills')) {
            return $recommendations; // Return empty if no position or skills relationship
        }
        
        $positionRequiredSkills = $employee->position->skills ?? collect();

        foreach ($positionRequiredSkills as $requiredSkill) {
            $currentLevel = $employeeSkills[$requiredSkill->id] ?? 0;
            $requiredLevel = $requiredSkill->pivot->required_level ?? 3;

            if ($currentLevel < $requiredLevel) {
                // Find training programs that develop this skill
                $trainingPrograms = TrainingProgram::whereHas('skills', function($query) use ($requiredSkill) {
                    $query->where('skill_id', $requiredSkill->id);
                })->where('is_active', true)->get();

                foreach ($trainingPrograms as $program) {
                    $priority = $this->calculateSkillGapPriority($currentLevel, $requiredLevel);
                    
                    $recommendation = TrainingRecommendation::createForEmployee(
                        $employee->id,
                        $program->id,
                        'skill_gap',
                        "Skill gap identified: {$requiredSkill->name} (Current: {$currentLevel}, Required: {$requiredLevel})",
                        $priority,
                        [
                            'skill_id' => $requiredSkill->id,
                            'current_level' => $currentLevel,
                            'required_level' => $requiredLevel,
                            'gap_size' => $requiredLevel - $currentLevel
                        ]
                    );

                    $recommendations[] = $recommendation;
                }
            }
        }

        return $recommendations;
    }

    private function generatePerformanceRecommendations(Employee $employee)
    {
        $recommendations = [];
        
        // Get latest performance review
        $latestReview = $employee->performanceReviews()
            ->where('status', 'completed')
            ->orderBy('review_date', 'desc')
            ->first();

        if ($latestReview && $latestReview->overall_rating < 3) {
            // Find training programs that address development areas
            $developmentAreas = $latestReview->development_areas ?? [];
            
            foreach ($developmentAreas as $area) {
                $trainingPrograms = TrainingProgram::where('is_active', true)
                    ->where(function($query) use ($area) {
                        $query->whereJsonContains('learning_objectives', $area)
                              ->orWhere('title', 'like', "%{$area}%")
                              ->orWhere('description', 'like', "%{$area}%");
                    })
                    ->get();

                foreach ($trainingPrograms as $program) {
                    $priority = $this->calculatePerformancePriority($latestReview->overall_rating);
                    
                    $recommendation = TrainingRecommendation::createForEmployee(
                        $employee->id,
                        $program->id,
                        'performance_improvement',
                        "Performance improvement needed in: {$area} (Overall rating: {$latestReview->overall_rating}/5)",
                        $priority,
                        [
                            'performance_review_id' => $latestReview->id,
                            'overall_rating' => $latestReview->overall_rating,
                            'development_area' => $area
                        ]
                    );

                    $recommendations[] = $recommendation;
                }
            }
        }

        return $recommendations;
    }

    private function generateCareerRecommendations(Employee $employee)
    {
        $recommendations = [];
        
        // Get training programs relevant to career advancement
        $careerPrograms = TrainingProgram::where('is_active', true)
            ->where('type', 'certification')
            ->orWhere('difficulty_level', 'advanced')
            ->get();

        foreach ($careerPrograms as $program) {
            $recommendation = TrainingRecommendation::createForEmployee(
                $employee->id,
                $program->id,
                'career_development',
                "Career development opportunity: {$program->title}",
                60, // Medium priority for career development
                [
                    'program_type' => $program->type,
                    'difficulty_level' => $program->difficulty_level
                ]
            );

            $recommendations[] = $recommendation;
        }

        return $recommendations;
    }

    private function calculateSkillGapPriority(int $currentLevel, int $requiredLevel): int
    {
        $gap = $requiredLevel - $currentLevel;
        return min(100, 50 + ($gap * 15)); // Higher gap = higher priority
    }

    private function calculatePerformancePriority(float $rating): int
    {
        if ($rating <= 2) return 90; // High priority for poor performance
        if ($rating <= 2.5) return 75; // Medium-high priority
        return 60; // Medium priority
    }
}