<?php

namespace App\Http\Controllers;

use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('viewAny', User::class);
        
        $query = User::with('roles');

        // Organization scoping: org_admin can only see users from their organization
        // System admins can see all users
        if (!Auth::user()->isSystemAdmin()) {
            $query->where('organization_id', Auth::user()->organization_id)
                  ->where('is_system_admin', false); // Hide system admins from org_admin
        }

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->whereHas('roles', function($q) use ($request) {
                $q->where('id', $request->role);
            });
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(15);
        
        // Only show roles available to the current user's organization
        $roles = $this->getAvailableRoles();

        return view('users.index', compact('users', 'roles'));
    }

    public function create()
    {
        $this->authorize('create', User::class);
        
        $roles = $this->getAvailableRoles();
        return view('users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', User::class);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'roles' => 'array',
            'roles.*' => 'exists:roles,id',
            'phone' => 'nullable|string|max:20',
        ]);

        $validated['password'] = Hash::make($validated['password']);
        
        // Set organization_id for new users (org_admin can only create users in their org)
        if (!Auth::user()->isSystemAdmin()) {
            $validated['organization_id'] = Auth::user()->organization_id;
        }

        $user = User::create($validated);

        if ($request->has('roles')) {
            // Validate that org_admin can only assign roles they have access to
            $availableRoleIds = $this->getAvailableRoles()->pluck('id')->toArray();
            $requestedRoleIds = array_intersect($request->roles, $availableRoleIds);
            
            // Convert IDs to Role models
            $roles = Role::whereIn('id', $requestedRoleIds)->get();
            $user->assignRole($roles);
        }

        return redirect()->route('users.index')
            ->with('success', 'User created successfully!');
    }

    public function show(User $user)
    {
        $this->authorize('view', $user);
        
        // Load roles with their permissions for display
        $user->load(['roles.permissions', 'employee']);
        return view('users.show', compact('user'));
    }

    public function edit(User $user)
    {
        $this->authorize('update', $user);
        
        $roles = $this->getAvailableRoles();
        return view('users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {
        $this->authorize('update', $user);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore($user->id)],
            'roles' => 'array',
            'roles.*' => 'exists:roles,id',
            'phone' => 'nullable|string|max:20',
        ]);

        $user->update($validated);

        if ($request->has('roles') && Auth::user()->can('manage_user_roles')) {
            // Validate that org_admin can only assign roles they have access to
            $availableRoleIds = $this->getAvailableRoles()->pluck('id')->toArray();
            $requestedRoleIds = array_intersect($request->roles, $availableRoleIds);
            
            // Convert IDs to Role models
            $roles = Role::whereIn('id', $requestedRoleIds)->get();
            $user->syncRoles($roles);
        }

        return redirect()->route('users.index')
            ->with('success', 'User updated successfully!');
    }

    public function destroy(User $user)
    {
        $this->authorize('delete', $user);
        
        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully!');
    }

    public function updatePassword(Request $request, User $user)
    {
        $this->authorize('update', $user);
        
        $validated = $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('success', 'Password updated successfully!');
    }

    /**
     * Get roles available to the current user based on their permissions
     */
    private function getAvailableRoles()
    {
        // System admins can see all roles
        if (Auth::user()->isSystemAdmin()) {
            return Role::all();
        }

        // Org admins cannot see system admin roles and certain restricted roles
        return Role::whereNotIn('name', ['system_admin'])->get();
    }
}
