<?php

namespace App\Http\Middleware;

use App\Services\AuditService;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class AuditMiddleware
{
    /**
     * Handle an incoming request and log audit events
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Only audit authenticated requests
        if (!Auth::check()) {
            return $response;
        }

        // Skip if auditing is disabled
        if (!config('audit.enabled', true)) {
            return $response;
        }

        // Skip certain routes/methods
        if ($this->shouldSkipAudit($request)) {
            return $response;
        }

        // Log the request
        $this->logRequest($request, $response);

        return $response;
    }

    /**
     * Determine if the request should be skipped from auditing
     */
    private function shouldSkipAudit(Request $request): bool
    {
        // Skip GET requests to audit logs (to prevent infinite loops)
        if ($request->is('*/audit-logs*') && $request->isMethod('GET')) {
            return true;
        }

        // Skip asset requests
        if ($request->is('assets/*') || $request->is('storage/*')) {
            return true;
        }

        // Skip API health checks
        if ($request->is('health') || $request->is('ping')) {
            return true;
        }

        // Skip AJAX requests for real-time updates
        if ($request->ajax() && $request->isMethod('GET')) {
            return true;
        }

        return false;
    }

    /**
     * Log the request as an audit event
     */
    private function logRequest(Request $request, Response $response): void
    {
        $eventType = $this->getEventType($request);
        $description = $this->getDescription($request, $response);
        $severity = $this->getSeverity($request, $response);
        $category = $this->getCategory($request);

        $metadata = [
            'response_status' => $response->getStatusCode(),
            'request_size' => strlen($request->getContent()),
            'response_size' => strlen($response->getContent()),
            'execution_time' => microtime(true) - LARAVEL_START,
        ];

        // Add form data for POST/PUT requests (excluding sensitive fields)
        if (in_array($request->method(), ['POST', 'PUT', 'PATCH'])) {
            $formData = $request->except(['password', 'password_confirmation', '_token', '_method']);
            if (!empty($formData)) {
                $metadata['form_data'] = $formData;
            }
        }

        AuditService::log(
            eventType: $eventType,
            description: $description,
            metadata: $metadata,
            severity: $severity,
            category: $category
        );
    }

    /**
     * Get event type based on request
     */
    private function getEventType(Request $request): string
    {
        return match($request->method()) {
            'GET' => 'page_accessed',
            'POST' => 'data_created',
            'PUT', 'PATCH' => 'data_updated',
            'DELETE' => 'data_deleted',
            default => 'request_made'
        };
    }

    /**
     * Get human-readable description
     */
    private function getDescription(Request $request, Response $response): string
    {
        $method = $request->method();
        $path = $request->path();
        $status = $response->getStatusCode();

        $action = match($method) {
            'GET' => 'Accessed',
            'POST' => 'Created data via',
            'PUT', 'PATCH' => 'Updated data via',
            'DELETE' => 'Deleted data via',
            default => 'Made request to'
        };

        return "{$action} {$path} (HTTP {$status})";
    }

    /**
     * Get severity based on request and response
     */
    private function getSeverity(Request $request, Response $response): string
    {
        $status = $response->getStatusCode();

        // Error responses are higher severity
        if ($status >= 500) {
            return 'high';
        }

        if ($status >= 400) {
            return 'medium';
        }

        // DELETE operations are medium severity
        if ($request->isMethod('DELETE')) {
            return 'medium';
        }

        // Admin/system routes are medium severity
        if ($request->is('admin/*') || $request->is('system-admin/*')) {
            return 'medium';
        }

        return 'low';
    }

    /**
     * Get category based on request
     */
    private function getCategory(Request $request): string
    {
        $path = $request->path();

        if (str_contains($path, 'login') || str_contains($path, 'logout') || str_contains($path, 'auth')) {
            return 'authentication';
        }

        if (str_contains($path, 'role') || str_contains($path, 'permission')) {
            return 'authorization';
        }

        if (str_contains($path, 'system-admin')) {
            return 'system_admin';
        }

        if ($request->isMethod('GET')) {
            return 'data_access';
        }

        return 'data_modification';
    }
}