<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckOnboardingStatus
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();

        // Skip for system admins
        if ($user && $user->is_system_admin) {
            return $next($request);
        }

        // Skip if no organization (shouldn't happen, but just in case)
        if (!$user || !$user->organization) {
            return $next($request);
        }

        $organization = $user->organization;

        // Whitelist routes that don't require onboarding completion
        $whitelistedRoutes = [
            'verification.notice',
            'verification.verify',
            'verification.verify.otp',
            'verification.resend',
            'verification.send',
            'onboarding.*',
            'organization.payments.*',
            'mpesa.callback',
            'logout',
            'password.*',
            'profile.*',
        ];

        foreach ($whitelistedRoutes as $pattern) {
            if ($request->routeIs($pattern)) {
                return $next($request);
            }
        }

        // Check if email needs verification
        if ($user->must_verify_email && !$user->hasVerifiedEmail()) {
            // Only redirect if not already on a verification route
            if (!$request->routeIs('verification.*') && !$request->routeIs('onboarding.*')) {
                return redirect()->route('verification.notice');
            }
            // If on onboarding route but email not verified, redirect to verification
            if ($request->routeIs('onboarding.*') && !$request->routeIs('onboarding.index')) {
                return redirect()->route('verification.notice');
            }
        }

        // Check if onboarding is complete
        if ($organization->onboarding_step !== 'completed') {
            // Skip if email verification is pending
            if ($organization->onboarding_step === 'pending_verification') {
                if (!$request->routeIs('verification.*') && !$request->routeIs('onboarding.index')) {
                    return redirect()->route('verification.notice');
                }
            } else {
                // For other onboarding steps, redirect to onboarding
                if (!$request->routeIs('onboarding.*') && !$request->routeIs('verification.*') && !$request->routeIs('organization.payments.*')) {
                    return redirect()->route('onboarding.index');
                }
            }
        }

        return $next($request);
    }
}
