<?php

namespace App\Http\Middleware;

use App\Services\UsageTrackingService;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckQuota
{
    protected $usageTrackingService;

    public function __construct(UsageTrackingService $usageTrackingService)
    {
        $this->usageTrackingService = $usageTrackingService;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     * @param  string  $metric  The metric to check (employees, users, departments, storage)
     */
    public function handle(Request $request, Closure $next, string $metric): Response
    {
        $user = $request->user();

        // System admins bypass quota checks
        if ($user && $user->is_system_admin) {
            return $next($request);
        }

        // Check if user has organization
        if (!$user || !$user->organization_id) {
            return redirect()->back()->with('error', 'No organization found');
        }

        $organization = $user->organization;
        $quota = $this->usageTrackingService->checkQuota($organization, $metric);

        // Check if quota is exceeded
        if ($quota['exceeded']) {
            $plan = $organization->currentPlan();
            $planName = $plan ? $plan->name : 'current plan';

            return redirect()->back()->with('error', 
                "You have reached your {$metric} limit ({$quota['limit']}) for the {$planName}. " .
                "Please upgrade your plan to add more {$metric}."
            );
        }

        // Check if approaching limit (90%)
        if (!($quota['unlimited'] ?? false) && $quota['percentage'] >= 90) {
            session()->flash('warning', 
                "You are using {$quota['percentage']}% of your {$metric} quota. " .
                "Consider upgrading your plan soon."
            );
        }

        return $next($request);
    }
}
