<?php

namespace App\Http\Middleware;

use App\Services\SecurityMonitoringService;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SecureFileUpload
{
    protected $securityMonitor;

    public function __construct(SecurityMonitoringService $securityMonitor)
    {
        $this->securityMonitor = $securityMonitor;
    }

    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Only process requests with file uploads
        if (!$request->hasFile()) {
            return $next($request);
        }

        $user = auth()->user();
        if (!$user) {
            return response()->json(['error' => 'Authentication required'], 401);
        }

        // Check each uploaded file
        foreach ($request->allFiles() as $key => $files) {
            $files = is_array($files) ? $files : [$files];
            
            foreach ($files as $file) {
                if (!$file->isValid()) {
                    continue;
                }

                $security = $this->securityMonitor->monitorFileUpload($file, $user);
                
                if (!$security['safe']) {
                    $this->securityMonitor->logSecurityEvent('unsafe_file_upload_blocked', [
                        'user_id' => $user->id,
                        'filename' => $file->getClientOriginalName(),
                        'issues' => $security['issues'],
                    ], 'warning');

                    return back()->withErrors([
                        $key => $this->getFileErrorMessage($security['issues'])
                    ]);
                }
            }
        }

        return $next($request);
    }

    /**
     * Get user-friendly error message for file issues
     */
    private function getFileErrorMessage(array $issues): string
    {
        $messages = [
            'file_too_large' => 'File size exceeds the maximum allowed limit (10MB).',
            'invalid_file_type' => 'File type is not allowed. Please upload a valid file.',
            'invalid_mime_type' => 'File format is not supported.',
        ];

        $errorMessages = [];
        foreach ($issues as $issue) {
            if (isset($messages[$issue])) {
                $errorMessages[] = $messages[$issue];
            }
        }

        return implode(' ', $errorMessages) ?: 'File upload failed security validation.';
    }
}