<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

abstract class BaseFormRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Handle a failed validation attempt.
     */
    protected function failedValidation(Validator $validator)
    {
        if ($this->expectsJson()) {
            throw new HttpResponseException(
                response()->json([
                    'message' => 'The given data was invalid.',
                    'errors' => $validator->errors()
                ], 422)
            );
        }

        parent::failedValidation($validator);
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Sanitize all string inputs
        $input = $this->all();
        $sanitized = $this->sanitizeInput($input);
        $this->replace($sanitized);
    }

    /**
     * Recursively sanitize input data
     */
    protected function sanitizeInput(array $input): array
    {
        foreach ($input as $key => $value) {
            if (is_array($value)) {
                $input[$key] = $this->sanitizeInput($value);
            } elseif (is_string($value)) {
                // Basic XSS protection
                $input[$key] = htmlspecialchars(strip_tags(trim($value)), ENT_QUOTES, 'UTF-8');
            }
        }

        return $input;
    }

    /**
     * Get common validation rules
     */
    protected function getCommonRules(): array
    {
        return [
            'name' => 'sometimes|string|max:255|regex:/^[a-zA-Z\s\-\.\']+$/',
            'email' => 'sometimes|email|max:255|filter:FILTER_SANITIZE_EMAIL',
            'phone' => 'sometimes|string|max:20|regex:/^[\+]?[0-9\-\(\)\s]+$/',
            'description' => 'sometimes|string|max:1000',
            'notes' => 'sometimes|string|max:2000',
        ];
    }

    /**
     * Get file upload validation rules
     */
    protected function getFileRules(): array
    {
        return [
            'file' => 'sometimes|file|max:10240|mimes:pdf,doc,docx,jpg,jpeg,png,gif',
            'image' => 'sometimes|image|max:5120|mimes:jpg,jpeg,png,gif',
            'document' => 'sometimes|file|max:10240|mimes:pdf,doc,docx,txt',
            'csv' => 'sometimes|file|max:5120|mimes:csv,txt',
            'excel' => 'sometimes|file|max:10240|mimes:xlsx,xls,csv',
        ];
    }
}