<?php

namespace App\Http\Requests\Employee;

use App\Http\Requests\BaseFormRequest;
use Illuminate\Validation\Rule;

class StoreEmployeeRequest extends BaseFormRequest
{
    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return array_merge($this->getCommonRules(), [
            'employee_id' => [
                'required',
                'string',
                'max:50',
                'regex:/^[A-Z0-9\-]+$/',
                Rule::unique('employees', 'employee_id')->where(function ($query) {
                    return $query->where('organization_id', auth()->user()->organization_id);
                })
            ],
            'first_name' => 'required|string|max:100|regex:/^[a-zA-Z\s\-\.\']+$/',
            'last_name' => 'required|string|max:100|regex:/^[a-zA-Z\s\-\.\']+$/',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('employees', 'email')->where(function ($query) {
                    return $query->where('organization_id', auth()->user()->organization_id);
                })
            ],
            'phone' => 'nullable|string|max:20|regex:/^[\+]?[0-9\-\(\)\s]+$/',
            'date_of_birth' => 'required|date|before:today|after:1900-01-01',
            'hire_date' => 'required|date|before_or_equal:today|after:1900-01-01',
            'department_id' => 'required|exists:departments,id',
            'position_id' => 'required|exists:positions,id',
            'salary' => 'required|numeric|min:0|max:999999999.99',
            'employment_type' => 'required|in:full_time,part_time,contract,intern',
            'status' => 'required|in:active,inactive,terminated',
            'national_id' => [
                'nullable',
                'string',
                'max:50',
                'regex:/^[A-Z0-9]+$/',
                Rule::unique('employees', 'national_id')->where(function ($query) {
                    return $query->where('organization_id', auth()->user()->organization_id);
                })
            ],
            'tax_number' => 'nullable|string|max:50|regex:/^[A-Z0-9\-]+$/',
            'bank_account' => 'nullable|string|max:50|regex:/^[0-9\-]+$/',
            'emergency_contact_name' => 'nullable|string|max:255|regex:/^[a-zA-Z\s\-\.\']+$/',
            'emergency_contact_phone' => 'nullable|string|max:20|regex:/^[\+]?[0-9\-\(\)\s]+$/',
            'address' => 'nullable|string|max:500',
            'photo' => 'nullable|image|max:2048|mimes:jpg,jpeg,png',
        ]);
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'employee_id.required' => 'Employee ID is required.',
            'employee_id.unique' => 'This employee ID is already taken in your organization.',
            'employee_id.regex' => 'Employee ID can only contain uppercase letters, numbers, and hyphens.',
            'first_name.required' => 'First name is required.',
            'first_name.regex' => 'First name can only contain letters, spaces, hyphens, dots, and apostrophes.',
            'last_name.required' => 'Last name is required.',
            'last_name.regex' => 'Last name can only contain letters, spaces, hyphens, dots, and apostrophes.',
            'email.required' => 'Email address is required.',
            'email.unique' => 'This email is already registered in your organization.',
            'date_of_birth.before' => 'Date of birth must be before today.',
            'date_of_birth.after' => 'Please enter a valid date of birth.',
            'hire_date.before_or_equal' => 'Hire date cannot be in the future.',
            'salary.min' => 'Salary must be a positive number.',
            'salary.max' => 'Salary amount is too large.',
            'national_id.unique' => 'This national ID is already registered in your organization.',
            'national_id.regex' => 'National ID can only contain uppercase letters and numbers.',
            'phone.regex' => 'Please enter a valid phone number.',
            'emergency_contact_phone.regex' => 'Please enter a valid emergency contact phone number.',
            'photo.image' => 'Photo must be an image file.',
            'photo.max' => 'Photo size cannot exceed 2MB.',
            'photo.mimes' => 'Photo must be a JPG, JPEG, or PNG file.',
        ];
    }

    /**
     * Configure the validator instance.
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            // Additional business logic validation
            if ($this->date_of_birth && $this->hire_date) {
                $age = now()->diffInYears($this->date_of_birth);
                if ($age < 16) {
                    $validator->errors()->add('date_of_birth', 'Employee must be at least 16 years old.');
                }
            }

            // Validate salary range based on position (if needed)
            if ($this->position_id && $this->salary) {
                // Add position-based salary validation if required
            }
        });
    }
}