<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class UpdateThemeRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->check() && auth()->user()->hasRole('org_admin');
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'base_theme' => 'required|string|in:default,system_admin,blue,green',
            
            // Color overrides (hex format)
            'primary_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'secondary_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'accent_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'success_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'warning_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'danger_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'info_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'light_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            'dark_color' => 'nullable|regex:/^#[0-9A-F]{6}$/i',
            
            // Sidebar colors (can be hex or rgba)
            'sidebar_bg_color' => 'nullable|string|max:50',
            'sidebar_text_color' => 'nullable|string|max:50',
            'sidebar_hover_color' => 'nullable|string|max:50',
            'sidebar_active_color' => 'nullable|string|max:50',
            
            // Fonts
            'primary_font' => 'nullable|string|max:100',
            'heading_font' => 'nullable|string|max:100',
            
            // Logo uploads
            'logo' => 'nullable|image|mimes:jpg,jpeg,png,svg|max:2048',
            'sidebar_logo' => 'nullable|image|mimes:jpg,jpeg,png,svg|max:2048',
            'header_logo' => 'nullable|image|mimes:jpg,jpeg,png,svg|max:2048',
            'favicon' => 'nullable|image|mimes:ico,png|max:512',
            'login_banner' => 'nullable|image|mimes:jpg,jpeg,png|max:4096',
            
            // Layout preferences
            'sidebar_collapsed_default' => 'nullable|boolean',
            'dark_mode_enabled' => 'nullable|boolean',
            
            // Custom CSS
            'custom_css' => 'nullable|string|max:51200', // 50KB
        ];
    }

    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'primary_color.regex' => 'Primary color must be in hex format (#RRGGBB)',
            'secondary_color.regex' => 'Secondary color must be in hex format (#RRGGBB)',
            'accent_color.regex' => 'Accent color must be in hex format (#RRGGBB)',
            'success_color.regex' => 'Success color must be in hex format (#RRGGBB)',
            'warning_color.regex' => 'Warning color must be in hex format (#RRGGBB)',
            'danger_color.regex' => 'Danger color must be in hex format (#RRGGBB)',
            'info_color.regex' => 'Info color must be in hex format (#RRGGBB)',
            'light_color.regex' => 'Light color must be in hex format (#RRGGBB)',
            'dark_color.regex' => 'Dark color must be in hex format (#RRGGBB)',
            'logo.max' => 'Logo must not exceed 2MB',
            'sidebar_logo.max' => 'Sidebar logo must not exceed 2MB',
            'header_logo.max' => 'Header logo must not exceed 2MB',
            'favicon.max' => 'Favicon must not exceed 512KB',
            'login_banner.max' => 'Login banner must not exceed 4MB',
            'custom_css.max' => 'Custom CSS must not exceed 50KB',
        ];
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Convert checkbox values to boolean
        $this->merge([
            'sidebar_collapsed_default' => $this->boolean('sidebar_collapsed_default'),
            'dark_mode_enabled' => $this->boolean('dark_mode_enabled'),
        ]);
        
        // Remove empty color values
        foreach (['primary_color', 'secondary_color', 'accent_color', 'success_color', 'warning_color', 'danger_color', 'info_color', 'light_color', 'dark_color'] as $color) {
            if ($this->input($color) === '') {
                $this->request->remove($color);
            }
        }
    }
}
