<?php

namespace App\Jobs;

use App\Models\AuditLog;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessAuditLog implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $maxExceptions = 3;
    public int $timeout = 30;

    public function __construct(
        private array $auditData
    ) {
        // Use a dedicated queue for audit logs to prevent blocking other jobs
        $this->onQueue('audit');
    }

    public function handle(): void
    {
        try {
            // Batch insert for better performance if multiple logs are queued
            AuditLog::create($this->auditData);
        } catch (\Exception $e) {
            Log::error('Failed to create audit log', [
                'error' => $e->getMessage(),
                'data' => $this->auditData
            ]);
            
            // Re-throw to trigger retry mechanism
            throw $e;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::critical('Audit log creation failed permanently', [
            'error' => $exception->getMessage(),
            'data' => $this->auditData
        ]);
    }
}