<?php

namespace App\Mail;

use App\Models\Application;
use App\Models\JobPosting;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use App\Mail\Concerns\UsesOrganizationMailSettings;

class ApplicationReceived extends Mailable implements ShouldQueue
{
    use Queueable, SerializesModels, UsesOrganizationMailSettings;

    public $application;
    public $jobPosting;

    /**
     * Create a new message instance.
     */
    public function __construct(Application $application, JobPosting $jobPosting)
    {
        $this->application = $application;
        $this->jobPosting = $jobPosting;
    }

    /**
     * Get the organization for mail configuration
     */
    protected function getOrganization()
    {
        return $this->jobPosting->organization ?? null;
    }

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        $fromAddress = $this->jobPosting->organization->use_custom_mail && $this->jobPosting->organization->mail_from_address
            ? $this->jobPosting->organization->mail_from_address
            : config('mail.from.address', 'noreply@hrms.com');
            
        $fromName = $this->jobPosting->organization->use_custom_mail && $this->jobPosting->organization->mail_from_name
            ? $this->jobPosting->organization->mail_from_name
            : $this->jobPosting->organization->name;

        return new Envelope(
            subject: "Application Received - {$this->jobPosting->title}",
            from: $fromAddress,
            replyTo: $fromAddress,
        );
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        return new Content(
            view: 'emails.application-received',
            with: [
                'application' => $this->application,
                'jobPosting' => $this->jobPosting,
                'organizationName' => $this->jobPosting->organization->name,
                'trackingUrl' => $this->application->tracking_url,
            ]
        );
    }

    /**
     * Get the attachments for the message.
     *
     * @return array<int, \Illuminate\Mail\Mailables\Attachment>
     */
    public function attachments(): array
    {
        return [];
    }
}