<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\HasTenantScope;
use Carbon\Carbon;

class Contract extends Model
{
    use HasFactory, SoftDeletes, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'contract_reference',
        'contract_type',
        'employment_type',
        'payment_frequency',
        'start_date',
        'end_date',
        'salary',
        'hourly_rate',
        'daily_rate',
        'overtime_rate_multiplier',
        'withholding_tax_applicable',
        'withholding_tax_rate',
        'statutory_deductions_applicable',
        'salary_structure',
        'working_hours_per_week',
        'standard_hours_per_day',
        'working_days_per_week',
        'working_schedule',
        'minimum_guaranteed_hours',
        'maximum_hours_per_week',
        'overtime_eligible',
        'benefits',
        'eligible_allowances',
        'eligible_statutory_benefits',
        'performance_bonus_eligible',
        'annual_increment_eligible',
        'increment_percentage',
        'notice_period_days',
        'auto_renewable',
        'last_renewed_date',
        'terms',
        'status',
        'document_path',
        'organization_id',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'last_renewed_date' => 'date',
        'salary' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'daily_rate' => 'decimal:2',
        'overtime_rate_multiplier' => 'decimal:2',
        'withholding_tax_rate' => 'decimal:4',
        'standard_hours_per_day' => 'decimal:2',
        'increment_percentage' => 'decimal:2',
        'working_hours_per_week' => 'integer',
        'working_days_per_week' => 'integer',
        'minimum_guaranteed_hours' => 'decimal:2',
        'maximum_hours_per_week' => 'decimal:2',
        'notice_period_days' => 'integer',
        'withholding_tax_applicable' => 'boolean',
        'statutory_deductions_applicable' => 'boolean',
        'overtime_eligible' => 'boolean',
        'performance_bonus_eligible' => 'boolean',
        'annual_increment_eligible' => 'boolean',
        'auto_renewable' => 'boolean',
        'working_schedule' => 'array',
        'eligible_allowances' => 'array',
        'eligible_statutory_benefits' => 'array',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function payrolls()
    {
        return $this->hasMany(Payroll::class);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeByEmploymentType($query, $type)
    {
        return $query->where('employment_type', $type);
    }

    public function scopeExpiringSoon($query, $days = 30)
    {
        return $query->where('end_date', '<=', now()->addDays($days))
                    ->where('status', 'active');
    }

    public function getIsExpiredAttribute()
    {
        return $this->end_date && $this->end_date < now();
    }

    public function getIsExpiringSoonAttribute()
    {
        return $this->end_date && $this->end_date <= now()->addDays(30);
    }

    public function getDaysUntilExpiryAttribute()
    {
        if (!$this->end_date) return null;
        return now()->diffInDays($this->end_date, false);
    }

    public function getEffectiveHourlyRateAttribute()
    {
        if ($this->hourly_rate) {
            return $this->hourly_rate;
        }

        if ($this->daily_rate && $this->standard_hours_per_day > 0) {
            return $this->daily_rate / $this->standard_hours_per_day;
        }

        if ($this->salary && $this->working_hours_per_week > 0) {
            // Calculate hourly rate from monthly salary
            $weeklyHours = $this->working_hours_per_week;
            $monthlyHours = $weeklyHours * 4.33; // Average weeks per month
            return $this->salary / $monthlyHours;
        }

        return 0;
    }

    public function getEffectiveDailyRateAttribute()
    {
        if ($this->daily_rate) {
            return $this->daily_rate;
        }

        if ($this->hourly_rate && $this->standard_hours_per_day > 0) {
            return $this->hourly_rate * $this->standard_hours_per_day;
        }

        if ($this->salary && $this->working_days_per_week > 0) {
            // Calculate daily rate from monthly salary
            $workingDaysPerMonth = $this->working_days_per_week * 4.33; // Average weeks per month
            return $this->salary / $workingDaysPerMonth;
        }

        return 0;
    }

    public function getOvertimeRateAttribute()
    {
        return $this->effective_hourly_rate * $this->overtime_rate_multiplier;
    }

    public function calculateBasicPayForPeriod($hoursWorked = null, $daysWorked = null)
    {
        switch ($this->payment_frequency) {
            case 'hourly':
                return $hoursWorked ? $hoursWorked * $this->effective_hourly_rate : 0;
                
            case 'daily':
                return $daysWorked ? $daysWorked * $this->effective_daily_rate : 0;
                
            case 'weekly':
                // For weekly contracts, calculate based on actual days/hours worked
                if ($daysWorked !== null) {
                    $weeklyRate = $this->salary / 4.33; // Convert monthly to weekly
                    return ($daysWorked / $this->working_days_per_week) * $weeklyRate;
                }
                return $this->salary / 4.33;
                
            case 'monthly':
            default:
                // For monthly salary, return full salary unless prorated
                return $this->salary;
        }
    }

    public function isEligibleForAllowance($allowanceType)
    {
        $eligibleAllowances = $this->eligible_allowances ?? [];
        return in_array($allowanceType, $eligibleAllowances) || empty($eligibleAllowances);
    }

    public function isEligibleForStatutoryBenefit($benefitType)
    {
        if (!$this->statutory_deductions_applicable) {
            return false;
        }

        $eligibleBenefits = $this->eligible_statutory_benefits ?? [];
        return in_array($benefitType, $eligibleBenefits) || empty($eligibleBenefits);
    }

    public function shouldApplyWithholdingTax()
    {
        return $this->withholding_tax_applicable && 
               in_array($this->employment_type, ['contract', 'freelance', 'casual']);
    }

    public function getContractDurationAttribute()
    {
        if (!$this->start_date || !$this->end_date) {
            return null;
        }

        return $this->start_date->diffForHumans($this->end_date, true);
    }

    public function getEmploymentTypeDisplayAttribute()
    {
        return match($this->employment_type) {
            'permanent' => 'Permanent Employee',
            'contract' => 'Contract Employee',
            'casual' => 'Casual Worker',
            'intern' => 'Intern',
            'freelance' => 'Freelancer',
            'part_time' => 'Part-time Employee',
            default => ucfirst($this->employment_type)
        };
    }

    public function getPaymentFrequencyDisplayAttribute()
    {
        return match($this->payment_frequency) {
            'monthly' => 'Monthly',
            'weekly' => 'Weekly',
            'daily' => 'Daily',
            'hourly' => 'Hourly',
            'project_based' => 'Project Based',
            default => ucfirst($this->payment_frequency)
        };
    }

    public function autoRenewIfEligible()
    {
        if ($this->auto_renewable && $this->is_expiring_soon && $this->status === 'active') {
            $this->update([
                'end_date' => $this->end_date->addYear(),
                'last_renewed_date' => now(),
            ]);
            
            return true;
        }
        
        return false;
    }
}
