<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\HasTenantScope;

class EmployeeAdvance extends Model
{
    use HasFactory, SoftDeletes, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'organization_id',
        'advance_number',
        'amount',
        'amount_repaid',
        'balance',
        'repayment_months',
        'monthly_deduction',
        'advance_date',
        'first_deduction_date',
        'status',
        'reason',
        'approved_by',
        'approved_at',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'amount_repaid' => 'decimal:2',
        'balance' => 'decimal:2',
        'monthly_deduction' => 'decimal:2',
        'advance_date' => 'date',
        'first_deduction_date' => 'date',
        'approved_at' => 'datetime',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Generate unique advance number
     */
    public static function generateAdvanceNumber(Employee $employee): string
    {
        $year = date('Y');
        $month = date('m');
        $count = static::whereYear('created_at', $year)->whereMonth('created_at', $month)->count() + 1;
        return "ADV-{$year}{$month}-{$employee->employee_code}-" . str_pad($count, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Calculate monthly deduction
     */
    public function calculateMonthlyDeduction(): float
    {
        return round($this->amount / $this->repayment_months, 2);
    }
}
